package gov.va.cpss.service;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_SITE_COUNT_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_STATEMENT_COUNT_KEY;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.stereotype.Service;

import gov.va.cpss.dao.PSDetailsDAO;
import gov.va.cpss.dao.PSPatientDAO;
import gov.va.cpss.dao.PSReceivedDAO;
import gov.va.cpss.dao.PSSiteDAO;
import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSReceived;
import gov.va.cpss.model.fps.PSSite;

/**
 * Service class for handling activities relating to processing FPS Data.
 * 
 * @author DNS   
 */
@Service
public class FpsService {

	private static final Logger logger = Logger.getLogger(FpsService.class.getCanonicalName());

	private ProcessStatusDAO processStatusDAO;
	private PSReceivedDAO psReceivedDAO;
	private PSSiteDAO psSiteDAO;
	private PSPatientDAO psPatientDAO;
	private PSDetailsDAO psDetailsDAO;

	public FpsService() {
	}

	public ProcessStatusDAO getProcessStatusDAO() {
		return processStatusDAO;
	}

	public void setProcessStatusDAO(ProcessStatusDAO processStatusDAO) {
		this.processStatusDAO = processStatusDAO;
	}

	public PSReceivedDAO getPsReceivedDAO() {
		return psReceivedDAO;
	}

	public void setPsReceivedDAO(PSReceivedDAO psReceivedDAO) {
		this.psReceivedDAO = psReceivedDAO;
	}

	public PSSiteDAO getPsSiteDAO() {
		return psSiteDAO;
	}

	public void setPsSiteDAO(PSSiteDAO psSiteDAO) {
		this.psSiteDAO = psSiteDAO;
	}

	public PSPatientDAO getPsPatientDAO() {
		return psPatientDAO;
	}

	public void setPsPatientDAO(PSPatientDAO psPatientDAO) {
		this.psPatientDAO = psPatientDAO;
	}

	public PSDetailsDAO getPsDetailsDAO() {
		return psDetailsDAO;
	}

	public void setPsDetailsDAO(PSDetailsDAO psDetailsDAO) {
		this.psDetailsDAO = psDetailsDAO;
	}

	public PSReceived startProcessFpsDataJob(final int batchRunId, final String fileName) {

		logger.info("Starting Process FPS Data Job");

		PSReceived pR = null;

		Integer processStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.INITIAL);

		if (processStatus != null) {
			Timestamp receivedDate = new Timestamp(Calendar.getInstance().getTime().getTime());

			pR = new PSReceived(batchRunId, processStatus, receivedDate, fileName);

			long receivedId = psReceivedDAO.save(pR);
			pR.setId(receivedId);
		} else {
			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.INITIAL);
		}

		return pR;
	}

	public boolean endProcessFpsDataJob(JobExecution execution, PSReceived psReceived) {

		boolean jobSuccess = false;

		if (execution == null) {

			logger.error("Job ended with null execution");
			errorProcessFpsDataJob(psReceived);

		} else if (execution.getExitStatus() != ExitStatus.COMPLETED) {

			if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
				logger.error("Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY));
			} else {
				logger.error("Job ended with unknown error: " + execution.getExitStatus());
			}
			errorProcessFpsDataJob(psReceived);
		} else {

			jobSuccess = successProcessFpsDataJob(execution, psReceived);

			// If unable to successfully save then treat as an error.
			if (!jobSuccess) {
				errorProcessFpsDataJob(psReceived);
			}
		}

		logger.info("End Process FPS Data Job");

		return jobSuccess;
	}

	private void errorProcessFpsDataJob(PSReceived psReceived) {

		// Be sure to delete any partial processed data.
		// Deleting the site entries will cascade delete through the child
		// tables.
		psSiteDAO.deleteByReceivedId(psReceived.getId());

		Integer processStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.ERROR);

		if (processStatus != null) {
			psReceived.setStatusId(processStatus);

			psReceivedDAO.updateStatus(psReceived);
		} else {
			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.ERROR);
		}
	}

	private boolean successProcessFpsDataJob(JobExecution execution, PSReceived psReceived) {

		boolean completeSuccess = true;

		if (execution.getExecutionContext().containsKey(TOTAL_SITE_COUNT_KEY)) {
			psReceived.setNumOfSite(execution.getExecutionContext().getLong(TOTAL_SITE_COUNT_KEY));
		} else {
			logger.error("Unable to obtain total site count from process results");
			completeSuccess = false;
		}

		if (execution.getExecutionContext().containsKey(TOTAL_STATEMENT_COUNT_KEY)) {
			psReceived.setNumOfStatement(execution.getExecutionContext().getLong(TOTAL_STATEMENT_COUNT_KEY));
		} else {
			logger.error("Unable to obtain total statement count from process results");
			completeSuccess = false;
		}

		if (completeSuccess) {

			// If successfully processed set the status to NEW to indicate ready
			// for processing.
			Integer processStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);

			if (processStatus != null) {

				psReceived.setStatusId(processStatus);
				psReceivedDAO.updateResults(psReceived);

			} else {

				logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.NEW);
				completeSuccess = false;
			}
		}

		return completeSuccess;
	}

	public Long getLastSiteIndex() {
		return psSiteDAO.getLastIndex();
	}

	public Long getLastPatientIndex() {
		return psPatientDAO.getLastIndex();
	}

	public Long getLastDetailsIndex() {
		return psDetailsDAO.getLastIndex();
	}

	public PSSite getPSSiteById(final long id) {
		return psSiteDAO.getById(id);
	}
	
	/**
	 * Returns a List of all PSSite records for NEW PSReceived records.
	 * 
	 * @return
	 */
	public List<PSSite> getPSSitesForNewPSReceived() {
		List<PSSite> psSites = new ArrayList<>();

		Integer newStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);

		if (newStatus != null) {
			psSites = psSiteDAO.getAllForPSReceivedStatusId(newStatus);
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.NEW);
		}

		return psSites;
	}

	public List<PSSite> getPSSiteForPSReceivedID(final long id) {
		return psSiteDAO.getAllByPSReceivedID(id);
	}

	public List<Long> getPSSiteIDListForPSReceivedID(final long id) {
		return psSiteDAO.getAllIDByPSReceivedID(id);
	}

	public List<PSPatient> getPSPatientListForPSSiteID(final long id) {
	
		// First get the PSSite from database.
		PSSite psSite = psSiteDAO.getById(id);
		
		return getPSPatientListForPSSite(psSite);
	}
	
	public List<PSPatient> getPSPatientListForPSSite(final PSSite psSite) {

		List<PSPatient> psPatientL = null;

		if (psSite != null) {

			// Then get all PSPatient from database.
			psPatientL = psPatientDAO.getAllByPSSiteID(psSite);

			// Then get all PSDetails from database.
			if (psPatientL != null) {

				List<PSDetails> psDetailsL = psDetailsDAO.getAllByPSSiteID(psSite);

				// Sort into a map (PSPatient.ID to List<PSDetails>) for the
				// list of PSDetails.
				Map<Long, List<PSDetails>> detailsQueryM = psDetailsL.stream()
						.collect(Collectors.groupingBy(w -> w.getPsPatientId()));

				// Set the appropriate details list in the PSPatient.
				for (PSPatient psPatient : psPatientL) {
					List<PSDetails> psDetailsMapEntry = detailsQueryM.get(psPatient.getId());
					// If no list then create an empty list.
					if (psDetailsMapEntry != null) {
						psPatient.setPsDetailsL(psDetailsMapEntry);
					} else {
						logger.error("Unrecoverable data error, details map is null for patient");
						return null;
					}
				}
			}

		}

		return psPatientL;
	}

	/**
	 * Get the NEW received data.
	 * 
	 * @return List of PSReceived ID associated with the NEW data records.
	 */
	public List<Long> getNewPSReceivedIDList() {

		List<Long> idL = new ArrayList<>();

		Integer newStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.NEW);

		if (newStatus != null) {
			idL = psReceivedDAO.getPSReceivedIDListByStatusID(newStatus);
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.NEW);
		}

		return idL;
	}

	/**
	 * Publish the NEW received data to PROCESSED status.
	 * 
	 * @return The number of received rows that were updated.
	 */
	public int processPSReceivedList(final List<Long> psReceivedL) {

		int updateCount = 0;

		Integer processedStatus = processStatusDAO.getStatusFromEnum(ProcessStatus.Status.PROCESSED);

		if (processedStatus != null) {

			for (Long id : psReceivedL) {
				updateCount += psReceivedDAO.updateStatusById(id, processedStatus);
			}
		} else {

			logger.error("Unable to obtain status mapping for: " + ProcessStatus.Status.PROCESSED);
		}

		return updateCount;
	}

	/**
	 * Save an PSSite and update the database auto generated ID.
	 * 
	 * @param site
	 *            The site record.
	 * @return The updated site populated with the auto generated ID.
	 */
	public void saveSite(PSSite site) {

		psSiteDAO.save(site);
	}

	/**
	 * Save an PSPatient and update the database auto generated ID.
	 * 
	 * @param patient
	 *            The patient record.
	 * @return The updated patient populated with the auto generated ID.
	 */
	public void savePatient(PSPatient patient) {

		psPatientDAO.save(patient);
	}
}
