package gov.va.cpss.model.apps;

import static gov.va.cpss.model.ps.Constants.DATE_FORMAT_STRING;
import static gov.va.cpss.model.ps.Constants.YEAR_FORMAT_STRING;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
//import java.time.LocalDate;
//import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;

import gov.va.cpss.model.ps.RecordType;
import gov.va.cpss.model.ps.Site;

/**
 * Model class for APSSite.
 * 
 * @author Yiping Yao
 * @version 1.0 - 20161019
 */
@SuppressWarnings("nls")
public class APSSite extends Site implements Serializable, APSRecord<RecordType>
{
    /**
     * Generated Serial Version UID
     */
    private static final long serialVersionUID = -4273145762269504671L;

    // Constants - Field lengths
    public static final int MAX_MSG_NUM_LENGTH = 10;
    public static final int MAX_TOT_MSG_NUM_LENGTH = MAX_MSG_NUM_LENGTH;
    public static final int MAX_FACILITY_NUM_LENGTH = 5;
    public static final int MAX_FACILITY_PHONE_NUM_LENGTH = 20;
    public static final int MAX_TOTAL_PATIENT_LENGTH = 4;

    // Properties
    private long apsReceivedId;
    private int totalPatient;
    private String totalPatientStr;
    private Date statementDate;
    private String statementDateStr;

    //
    // Constructors
    //
    public APSSite()
    {
        // Default constructor
    }

    public APSSite(final long inId)
    {
        this.id = inId;
    }

    //
    // Getters / Setters
    //
    /**
     * @return the apsReceivedId
     */
    public long getApsReceivedId()
    {
        return this.apsReceivedId;
    }

    /**
     * @param inApsReceivedId the apsReceivedId to set
     */
    public void setApsReceivedId(long inApsReceivedId)
    {
        this.apsReceivedId = inApsReceivedId;
    }

    /**
     * @return the totalPatient
     */
    public int getTotalPatient()
    {
        return this.totalPatient;
    }

    /**
     * @param inTotalPatient the totalPatient to set
     */
    public void setTotalPatient(int inTotalPatient)
    {
        this.totalPatient = inTotalPatient;
    }

    /**
     * @return the totalPatientStr
     */
    public String getTotalPatientStr()
    {
        return this.totalPatientStr;
    }

    /**
     * @param inTotalPatientStr the totalPatientStr to set
     */
    public void setTotalPatientStr(String inTotalPatientStr)
    {
        this.totalPatientStr = inTotalPatientStr;

        if (this.totalPatientStr != null &&
            StringUtils.isNumeric(this.totalPatientStr))
        {
            this.totalPatient = Integer.parseInt(this.totalPatientStr);
        }
    }

    /**
     * @return the statementDate
     */
    public Date getStatementDate()
    {
        return this.statementDate;
    }

    /**
     * @param inStatementDate the statementDate to set
     */
    public void setStatementDate(Date inStatementDate)
    {
        this.statementDate = inStatementDate;
    }

    /**
     * @return the statementDateStr
     */
    public String getStatementDateStr()
    {
        return this.statementDateStr;
    }

    /**
     * @param inStatementDateStr the statementDateStr to set
     */
    public void setStatementDateStr(String inStatementDateStr)
    {
        this.statementDateStr = inStatementDateStr;

        if (this.statementDateStr != null &&
            StringUtils.isNumeric(this.statementDateStr))
        {
            try
            {
                this.statementDate = new SimpleDateFormat(YEAR_FORMAT_STRING, Locale.ENGLISH).parse(this.statementDateStr);
            }
            catch (ParseException e)
            {
                e.printStackTrace();
            }
        }
    }

    /**
     * Return false if there are any validation errors and true if there are none.
     */
    @Override
    public boolean isValid()
    {
        boolean isValid = true;
        StringBuffer strBuffer = new StringBuffer();

        if (StringUtils.isEmpty(this.seqNumStr) ||
            this.seqNumStr.length() > MAX_MSG_NUM_LENGTH ||
            !StringUtils.isNumeric(this.seqNumStr))
        {
            strBuffer.append("seqNum is invalid (PS002). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.totSeqNumStr) ||
            this.totSeqNumStr.length() > MAX_TOT_MSG_NUM_LENGTH ||
            !StringUtils.isNumeric(this.totSeqNumStr))
        {
            strBuffer.append("totSeqNum is invalid (PS003). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.facilityNum) ||
            this.facilityNum.length() > MAX_FACILITY_NUM_LENGTH)
        {
            strBuffer.append("facilityNum is invalid (PS004). ");
            isValid = false;
        }

        // Optional
        if ( !StringUtils.isEmpty(this.facilityPhoneNum) &&
             (this.facilityPhoneNum.length() > MAX_FACILITY_PHONE_NUM_LENGTH) )
        {
            strBuffer.append("facilityPhoneNum is invalid (PS005). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.totalPatientStr) ||
            this.totalPatientStr.length() > MAX_TOTAL_PATIENT_LENGTH ||
            !StringUtils.isNumeric(this.totalPatientStr))
        {
            strBuffer.append("totalPatient is invalid (PS006). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.statementDateStr) ||
            this.statementDateStr.length() > MAX_YEAR_LENGTH ||
            !StringUtils.isNumeric(this.statementDateStr))
        {
            strBuffer.append("statementDate (year) is invalid (PS007). ");
            isValid = false;
        }

        if (StringUtils.isEmpty(this.processDateStr) ||
            this.processDateStr.length() != MAX_DATE_LENGTH)
        {
            strBuffer.append("processDate is invalid (PS008). ");
            isValid = false;
        }

        // Catch an exception if the format is incorrect.
        try
        {
            new SimpleDateFormat(DATE_FORMAT_STRING).parse(this.processDateStr);
        }
        catch(@SuppressWarnings("unused") ParseException e)
        {
            strBuffer.append("processDate is invalid (PS008). ");
            isValid = false;
        }

        if (!isValid)
        {
            this.message = strBuffer.toString();
        }
            
        return isValid;
    }

    @Override
    public String toString()
    {
        return String.format("APS Site (%s) [seqNum=%s, totSeqNum=%s, facilityNum=%s, facilityPhoneNum=%s, totalPatient=%s, statementDate=%s, processDate=%s]" + " Message: %s",
                             this.type, this.seqNumStr, this.totSeqNumStr, this.facilityNum, this.facilityPhoneNum, this.totalPatientStr, this.statementDateStr, this.processDateStr,
                             (this.message == null ? "none" : this.message));
    }
}
