package gov.va.cpss.model;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.util.UnknownFormatConversionException;

import org.apache.commons.lang.math.NumberUtils;

/**
 * Model for representing Dollar fields for text file transfers from/to AITC.
 * Text format is in the form of a positive integer number with two decimal
 * places implied followed by + or - indicating the sign.
 * 
 * Ex.
 * 
 * 99999- represents -$999.99 111111+ represents $1111.11
 * 
 * @author Brad Pickle
 *
 */
@SuppressWarnings("nls")
public class AITCDollar implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Enumeration for representing the sign character.
	 * 
	 * @author Brad Pickle
	 *
	 */
	public enum Sign {
		PLUS(1, '+'), MINUS(-1, '-');

		private int sign;
		private char value;

		private Sign(int sign, char value) {
			this.sign = sign;
			this.value = value;
		}

		public int getSign() {
			return sign;
		}

		public char getValue() {
			return value;
		}

		/**
		 * Get the Sign enum for the given String. Sign is looked for in the
		 * first position of the String.
		 * 
		 * @param c
		 *            Sign String such as "+" or "-"
		 * @return Sign for the given String.
		 * @throws UnknownFormatConversionException
		 *             If c is null or empty or if valid sign is not found in first position.
		 */
		public static Sign from(final String c) throws UnknownFormatConversionException {
			if ((c == null) || c.isEmpty()) {
				throw new UnknownFormatConversionException("Invalid string value (" + c + ") for AITCDollar.Sign.");
			}
			return from(c.charAt(0));
		}

		/**
		 * Get the Sign enum for the given AITC Dollar String. Sign is looked for in
		 * the last position of the String as expected in AITC Dollar format.
		 * 
		 * @param c
		 *            AITC Dollar String
		 * @return Sign for the given AITC Dollar String
		 * @throws UnknownFormatConversionException
		 *             If c is null or empty or if valid sign is not found in last position.
		 */
		public static Sign fromEnd(final String c) throws UnknownFormatConversionException {
			if ((c == null) || c.isEmpty()) {
				throw new UnknownFormatConversionException("Invalid string value (" + c + ") for AITCDollar.Sign.");
			}
			return from(c.charAt(c.length() - 1));
		}

		/**
		 * Get the Sign enum for the given char.
		 * 
		 * @param c
		 *            char representing Sign
		 * @return Sign for the given char.
		 * @throws UnknownFormatConversionException
		 *             If sign is not valid.
		 */
		public static Sign from(final char c) throws UnknownFormatConversionException {

			if (c == PLUS.getValue())
				return PLUS;
			if (c == MINUS.getValue())
				return MINUS;

			throw new UnknownFormatConversionException("Invalid char value (" + c + ") for AITCDollar.Sign.");
		}
		
		public static Sign from(final double d) {
			return (d < 0) ? MINUS : PLUS;
		}
	}

	private double amount;
	
	private Sign sign;

	public AITCDollar(double amount) {
		this.amount = amount;
		this.sign = Sign.from(amount);
	}

	public double getDouble() {
		return amount;
	}
	
	public String toPaddedString(final int size) {
		
		int digitLength = size - 1;
		
    	StringBuilder sb = new StringBuilder(new DecimalFormat("0.00").format(Math.abs(amount)));
    	
    	// Get absolute long representation value.
		final String absLongS = sb.deleteCharAt(sb.length() - 3).toString();
		long longValue = Long.valueOf(absLongS);

		// Verify size is large enough. If not then the number is invalid and
		// throw and exception.
		if (absLongS.length() > digitLength) {
			throw new IndexOutOfBoundsException("AITC format requires larger size than specified. Size for "
					+ amount + " requires at least " + (absLongS.length() + 1) + " digits.");
		}

		// Preformat the aitc value with specified digit length by prepadding
		// with leading zeroes.
		String aitcValue = String.format("%%0%dd", digitLength);
		aitcValue = String.format(aitcValue, longValue);

		aitcValue += sign.value;
		
		return aitcValue;
	}

	@Override
	public String toString() {
		return String.format("%.2f", Math.abs(amount)).replace(".", "").replaceFirst("^0*(\\d{1})", "$1") + sign.value;
	}

	/**
	 * Create a new AITCDollar from the given AITC Dollar String.
	 * 
	 * @param aitcDollar AITC Dollar String
	 * @return New AITCDollar object initialized from the given String.
	 * @throws IllegalArgumentException If the given String is not a valid AITC Dollar String.
	 */
	public static AITCDollar fromString(final String aitcDollar) throws IllegalArgumentException {
		if (aitcDollar == null)
			throw new NumberFormatException("Cannot convert null to AITCDollar.");

		final String trimmed = aitcDollar.trim();
		final int length = trimmed.length();

		if (length < 2)
			throw new NumberFormatException("Invalid format for AITCDollar: " + trimmed);

		final Sign sign = Sign.fromEnd(trimmed);

		String doubleString;
		if (length == 2) {
			doubleString = ".0" + trimmed.substring(0, 1);
		} else {
			doubleString = trimmed.substring(0, length - 3) + "." + trimmed.substring(length - 3, length - 1);
		}

		return new AITCDollar(Math.abs(Double.valueOf(doubleString).doubleValue()) * sign.getSign());
	}

    /**
     * Checks to see if a number (amount) is in the form of an AITCDollar number.
     */
    public static boolean isAITCDollarFormat(String amount)
    {
        char lastChar = amount.charAt(amount.length() - 1);
        String shortAmount = amount;

        if (!NumberUtils.isDigits(String.valueOf(lastChar)))
        {
            if(lastChar == Sign.PLUS.getValue() || lastChar == Sign.MINUS.getValue())
            {
                shortAmount = amount.substring(0, amount.length() - 1);
            }
            else
            {
                return false;
            }
        }

        //After removing the plus and minus signs, only numbers should be left over.
        return NumberUtils.isDigits(shortAmount);
    }

    /**
     * Create a new AITCDollar from the given AITC Dollar String without the +/- sign.
     * (For APPS)
     * 
     * Yiping Yao - 20170314
     * 
     * @param aitcDollar AITC Dollar String
     * @return New AITCDollar object initialized from the given String.
     * @throws IllegalArgumentException If the given String is not a valid AITC Dollar String.
     */
    public static AITCDollar fromStringWithoutSign(final String aitcDollar) throws IllegalArgumentException
    {
        if (aitcDollar == null)
        {
            throw new NumberFormatException("Cannot convert null to AITCDollar.");
        }

        final String trimmed = aitcDollar.trim();
        final int length = trimmed.length();

        if (length < 1)
        {
            throw new NumberFormatException("Invalid format for AITCDollar: " + trimmed);
        }

        String doubleString;

        if (length == 1)
        {
            doubleString = ".0" + trimmed;
        }
        else
        {
            doubleString = trimmed.substring(0, length - 2) + "." + trimmed.substring(length - 2, length);
        }

        return new AITCDollar(Math.abs(Double.valueOf(doubleString).doubleValue()));
    }

    /**
     * Checks to see if a number (amount) is in the form of an AITCDollar number without the +/- sign.
     * (For APPS)
     * 
     * Yiping Yao - 20170314
     * 
     */
    public static boolean isAITCDollarFormatWithoutSign(String amount)
    {
        return NumberUtils.isDigits(amount);
    }
}
