package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_MAX_RECORDS_PER_SITE;

import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PS1;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.model.fps.PSSiteStmt;
import gov.va.cpss.model.sendcbs.PSDate;

public class SendCBSWritePSRecordSource {

	@SuppressWarnings("unused")
	private Logger logger;

	private SendCBSRuntimeState sendCBSRuntimeState;

	private int currentPSSiteIndex = 0;
	private int currentCBSStmtIndex = 0;
	private int currentCBSSiteStmtIndex = 0;
	private int currentCBSSiteTransIndex = 0;

	private boolean psWritten = false;
	private boolean phWritten = false;
	private boolean pvWritten = false;
	//Added for consolidated statement;
	private boolean multisiteflag =false;

	public SendCBSWritePSRecordSource() {
		logger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public SendCBSRuntimeState getSendCBSRuntimeState() {
		return sendCBSRuntimeState;
	}

	public void setSendCBSRuntimeState(SendCBSRuntimeState sendCBSRuntimeState) {
		this.sendCBSRuntimeState = sendCBSRuntimeState;
	}

	public Map<String, List<CBSStmt>> getCbsListsBySite() {
		return sendCBSRuntimeState.getCBSListsBySite();
	}

	public Map<String, List<CBSStmt>> getSbsListsBySite() {
		return sendCBSRuntimeState.getSBSListsBySite();
	}

	public List<PSSite> getPsSitesList() {
		return sendCBSRuntimeState.getPSSitesList();
	}

	public void setCbsListsBySite(Map<String, List<CBSStmt>> cbsListsBySite) {
		this.sendCBSRuntimeState.setCbsListsBySite(cbsListsBySite);
	}
	
	public void setSbsListsBySite(Map<String, List<CBSStmt>> sbsListsBySite) {
		this.sendCBSRuntimeState.setSbsListsBySite(sbsListsBySite);
	}
	
	public void setPsSitesList(List<PSSite> psSitesList) {
		this.sendCBSRuntimeState.setPsSitesList(psSitesList);
	}
	
	public PSRecord nextPSRecord() {
		return nextPSRecord(false);
	}
	
	public PSRecord nextPSRecord(final boolean onlyPatientFlag) {

		// Return null if all PS records processed
		if (!psRecordsRemaining())
			return null;

		if (!psWritten)
			return nextPSSite();

		if (!phWritten)
			return nextPSPatientOrPS1(onlyPatientFlag);

		if (!pvWritten)
			return nextPSSiteStmt();

		return nextPSDetails();
	}


	private boolean psRecordsRemaining() {
		return (currentPSSiteIndex < sendCBSRuntimeState.getPSSitesList().size());
	}

	private PSSite nextPSSite() {
		// Mark the PS as written
		psWritten = true;

		PSSite psSite = getCurrentPSSite();

		if (psSite != null) {
			psSite.setStatementDate(getPSDate(psSite.getStatementDate()));
			psSite.setProcessDate(getPSDate(psSite.getProcessDate()));
		}

		return psSite;
	}

	protected PSRecord nextPSPatientOrPS1(final boolean onlyPatientFlag) {
		if (onlyPatientFlag || (getCurrentCBSSiteStmtList().size() > 1))
			return nextPSPatient();
		else
			return nextPS1();
	}

	private PS1 nextPS1() {

		CBSStmt cbsStmt = getCurrentCBSStmt();

		CBSSiteStmt primarySite = getPrimarySite(cbsStmt);
		CBSSitePatient cbsPrimarySitePatient = primarySite.getSitePatient();

		PS1 ps1 = new PS1();
		ps1.setType(PSRecord.DataType.P1);

		// From primary site patient CBSSitePatient record
		ps1.setPatientAccount(cbsPrimarySitePatient.getOldAcntNum());
		ps1.setIcnNumber(cbsPrimarySitePatient.getIcn());

		// From CBSStmt record
		ps1.setAcntNumDisp(cbsStmt.getAcntNumDisp());

		// Expecting cbsStmt.getLateStmtMsg to by Y or N
		ps1.setLateSiteFlag(BooleanChar.from(cbsStmt.getLateStmtMsg()));

		// This completes the current CBSStmt, advance to the next CBSStmt or
		// PSSite record.
		advanceCBSStmt();

		return ps1;
	}

	protected PSPatient nextPSPatient() {
		CBSStmt cbsStmt = getCurrentCBSStmt();

		CBSSiteStmt primarySite = getPrimarySite(cbsStmt);
		CBSSitePatient cbsPrimarySitePatient = primarySite.getSitePatient();

		CBSSiteStmt primaryAddressSite = getPrimaryAddressSite(cbsStmt);
		CBSSitePatient cbsPrimaryAddressSitePatient = primaryAddressSite.getSitePatient();

		PSPatient psPatient = new PSPatient();
		psPatient.setType(PSRecord.DataType.PH);

		psPatient.setPsSite(new PSSite());

		// From primary site patient CBSSitePatient record
		psPatient.setPatientAccount(cbsPrimarySitePatient.getOldAcntNum());
		psPatient.setPatientLastName(cbsPrimarySitePatient.getLastName());
		psPatient.setPatientFirstName(cbsPrimarySitePatient.getFirstName());
		psPatient.setPatientMiddleName(cbsPrimarySitePatient.getMiddleName());

		// From primary address site patient CBSSitePatient record
		psPatient.setAddress1(cbsPrimaryAddressSitePatient.getAddress1());
		psPatient.setAddress2(cbsPrimaryAddressSitePatient.getAddress2());
		psPatient.setAddress3(cbsPrimaryAddressSitePatient.getAddress3());
		psPatient.setCity(cbsPrimaryAddressSitePatient.getCity());
		psPatient.setState(cbsPrimaryAddressSitePatient.getState());
		psPatient.setZipCode(cbsPrimaryAddressSitePatient.getZipCode());
		psPatient.setCountryName(cbsPrimaryAddressSitePatient.getCountry());

		// From CBSStmt record
		psPatient.setAmountDue(cbsStmt.getAmountDue());
		psPatient.setPrevBalance(cbsStmt.getPrevBalance());
		psPatient.setTotalCharges(cbsStmt.getTotalCharges());
		psPatient.setTotalCredits(cbsStmt.getTotalCredits());
		psPatient.setNewBalance(cbsStmt.getNewBalance());

		// From primary site CBSSiteStmt.
		// For specialNotes, first 80 characters goes to specialNotes, next 40
		// goes to specialNotesForLateSites1,
		// last 40 goes to specialNotesForLateSites2
		String specialNotes = (primarySite.getSpecialNotes() == null) ? "" : primarySite.getSpecialNotes();
		psPatient.setSpecialNotes((specialNotes.length() >= 80) ? specialNotes.substring(0, 80) : specialNotes);
		psPatient.setSpecialNotesForLateSites1((specialNotes.length() > 80)
				? ((specialNotes.length() >= 120) ? specialNotes.substring(80, 120) : specialNotes.substring(80)) : "");
		psPatient.setSpecialNotesForLateSites2((specialNotes.length() > 120) ? specialNotes.substring(120) : "");

		// From primary site CBSSiteStmt record
		psPatient.setRightsObligationParagraphCodes(primarySite.getNoParaCdes());

		// Total number of nested CBSSiteStmt, CBSSiteTrans records in CBSStmt record
		psPatient.setNumOfLines(getNumOfLines(cbsStmt));

		// From primary site patient CBSSitePatient record
		psPatient.setDfnNumber(cbsPrimarySitePatient.getDfn());

		// From primary address site CBSSiteStmt record
		psPatient.setLargeFontInd(primaryAddressSite.getLargeFontInd());

		// From primary site patient CBSSitePatient record
		psPatient.setIcnNumber(cbsPrimarySitePatient.getIcn());

		// From primary site CBSSiteStmt record
		psPatient.setAddressFlag(primarySite.getArAddressFlag());
		psPatient.setLastBillPrepDate(getPSDate(primarySite.getLastBillPrepDate()));

		// From CBSStmt record
		psPatient.setAcntNumDisp(cbsStmt.getAcntNumDisp());

		// Total number of nested CBSSiteStmt records in CBSStmt record
		psPatient.setNumConsolidatedSites(getNumConsolidatedSites(cbsStmt));

		// Expecting cbsStmt.getLateStmtMsg to by Y or N
		psPatient.setLateSiteFlag(BooleanChar.from(cbsStmt.getLateStmtMsg()));

		// Mark the PH as written
		phWritten = true;

		return psPatient;
	}

	private PSSiteStmt nextPSSiteStmt() {
		CBSSiteStmt cbsSiteStmt = getCurrentCBSSiteStmt();
		CBSSitePatient cbsSitePatient = cbsSiteStmt.getSitePatient();

		PSSiteStmt psSiteStmt = new PSSiteStmt();
		psSiteStmt.setType(PSRecord.DataType.PV);

		// From CBSSiteStmt record
		psSiteStmt.setFacilityNum(cbsSiteStmt.getStationNum());
		psSiteStmt.setStationTotalAmount(cbsSiteStmt.getNewBalance());

		// Total number of nested CBSSitePatient records in CBSSiteStmt record
		psSiteStmt.setNumOfLines(getNumOfLines(cbsSiteStmt));

		// From CBSSitePatient record
		psSiteStmt.setPatientAccount(cbsSitePatient.getOldAcntNum());
		psSiteStmt.setDfnNumber(cbsSitePatient.getDfn());
		psSiteStmt.setIcnNumber(cbsSitePatient.getIcn());

		// Mark the PV as written
		pvWritten = true;

		return psSiteStmt;
	}

	private PSDetails nextPSDetails() {
		CBSSiteTrans cbsSiteTrans = getCurrentCBSSiteTrans();
		CBSSitePatient cbsSitePatient = getCurrentCBSSiteStmt().getSitePatient();

		PSDetails psDetails = new PSDetails();
		psDetails.setType(PSRecord.DataType.PD);

		// From CBSSiteTrans record
		psDetails.setDatePosted(getPSDate(cbsSiteTrans.getDatePosted()));
		psDetails.setTransDesc(cbsSiteTrans.getTransDesc());
		psDetails.setTransAmount(cbsSiteTrans.getTransAmount());
		psDetails.setReferenceNum(cbsSiteTrans.getReferenceNum());

		// From CBSSitePatient record
		psDetails.setIcnNumber(cbsSitePatient.getIcn());

		// From CBSSiteTrans record
		psDetails.setSeqNum(cbsSiteTrans.getOrderNum());

		// Advance to the next CBSSiteTrans, CBSSiteStmt, CBSStmt or PSSite
		// record
		advanceCBSSiteTrans();

		return psDetails;
	}

	private void advanceCBSSiteTrans() {
		// Advance to next CBSSiteTrans
		currentCBSSiteTransIndex++;

		if (getCurrentCBSSiteTransList().size() > currentCBSSiteTransIndex)
			return;

		// No more CBSSiteTrans records. Advance to next CBSSiteStmt
		advanceCBSSiteStmt();
	}

	private void advanceCBSSiteStmt() {
		// Advance to next CBSSiteStmt
		currentCBSSiteStmtIndex++;

		currentCBSSiteTransIndex = 0;
		pvWritten = false;

		if (getCurrentCBSSiteStmtList().size() > currentCBSSiteStmtIndex)
			return;

		// No more CBSSiteStmt records. Advance to next CBSStmt
		advanceCBSStmt();
	}

	private void advanceCBSStmt() {
		// Advance to next CBSStmt
		++currentCBSStmtIndex;

		currentCBSSiteTransIndex = 0;
		currentCBSSiteStmtIndex = 0;
		pvWritten = false;
		phWritten = false;

		// Check if there are more statements to process.
		if (getCurrentCBSStmtList().size() > currentCBSStmtIndex) {

			// Only allowed MAX_RECORDS_PER_SITE number of statements per PS so
			// advance to next PS with split statement count if have reached the
			// maximum for the current PS.
			if ((currentCBSStmtIndex % CBS_MAX_RECORDS_PER_SITE) == 0) {
				advancePSSite(true);
			}

		} else {

			// No more CBSStmt records. Advance to next PSSite.
			advancePSSite(false);
		}
	}

	private void advancePSSite(final boolean split) {
		// Advance to next PSSite
		currentPSSiteIndex++;

		// If not splitting statements across different PS then reset statement
		// counter.
		if (!split) {
			currentCBSStmtIndex = 0;
		}

		currentCBSSiteStmtIndex = 0;
		currentCBSSiteTransIndex = 0;
		psWritten = false;
		phWritten = false;
		pvWritten = false;
	}

	private PSSite getCurrentPSSite() {
		return (sendCBSRuntimeState.getPSSitesList().size() > currentPSSiteIndex)
				? sendCBSRuntimeState.getPSSitesList().get(currentPSSiteIndex) : null;
	}

	private List<CBSStmt> getCurrentCBSStmtList() {
		List<CBSStmt> cbsStmtL = null;

		PSSite psSite = getCurrentPSSite();
		if (psSite != null) {
			if (psSite.getStatementType() == PSRecord.DataType.P1) {
				cbsStmtL = sendCBSRuntimeState.getSBSListsBySite().get(psSite.getFacilityNum());
			} else {
				cbsStmtL = sendCBSRuntimeState.getCBSListsBySite().get(psSite.getFacilityNum());
			}
		}

		return cbsStmtL;
	}

	private CBSStmt getCurrentCBSStmt() {
		List<CBSStmt> cbsStmtList = getCurrentCBSStmtList();
		return ((cbsStmtList == null) || (cbsStmtList.size() <= currentCBSStmtIndex)) ? null
				: cbsStmtList.get(currentCBSStmtIndex);
	}

	private List<CBSSiteStmt> getCurrentCBSSiteStmtList() {
		CBSStmt cbsStmt = getCurrentCBSStmt();
		return (cbsStmt == null) ? null : cbsStmt.getSiteStmtL();
	}

	private CBSSiteStmt getCurrentCBSSiteStmt() {
		List<CBSSiteStmt> cbsSiteStmtList = getCurrentCBSSiteStmtList();
		return ((cbsSiteStmtList == null) || (cbsSiteStmtList.size() <= currentCBSSiteStmtIndex)) ? null
				: cbsSiteStmtList.get(currentCBSSiteStmtIndex);
	}

	private List<CBSSiteTrans> getCurrentCBSSiteTransList() {
		CBSSiteStmt cbsSiteStmt = getCurrentCBSSiteStmt();
		return (cbsSiteStmt == null) ? null : cbsSiteStmt.getSiteTransL();

	}

	private CBSSiteTrans getCurrentCBSSiteTrans() {
		List<CBSSiteTrans> cbsSiteTransList = getCurrentCBSSiteTransList();
		return ((cbsSiteTransList == null) || (cbsSiteTransList.size() <= currentCBSSiteTransIndex)) ? null
				: cbsSiteTransList.get(currentCBSSiteTransIndex);
	}

	private CBSSiteStmt getPrimarySite(CBSStmt cbsStmt) {
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			if (cbsSiteStmt.getIsPrimary().isTrue()) {
				return cbsSiteStmt;
			}
		}

		return cbsStmt.getSiteStmtL().get(0);
	}

	private CBSSiteStmt getPrimaryAddressSite(CBSStmt cbsStmt) {
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			if (cbsSiteStmt.getIsPrimaryAddress().isTrue()) {
				return cbsSiteStmt;
			}
		}

		return cbsStmt.getSiteStmtL().get(0);
	}

	private int getNumOfLines(CBSStmt cbsStmt) {
		int numOfLines = 0;
		for (CBSSiteStmt cbsSiteStmt : cbsStmt.getSiteStmtL()) {
			numOfLines += getNumOfLines(cbsSiteStmt) + 1;
		}
		return numOfLines;
	}

	private int getNumOfLines(CBSSiteStmt cbsSiteStmt) {
		List<CBSSiteTrans> siteTransL = cbsSiteStmt.getSiteTransL();
		return ((siteTransL == null) ? 0 : siteTransL.size());
	}

	private int getNumConsolidatedSites(CBSStmt cbsStmt) {
		return cbsStmt.getSiteStmtL().size();
	}

	private PSDate getPSDate(final Date date) {
		return (date == null) ? new PSDate(0L) : new PSDate(date.getTime());
	}

	public boolean isMultisiteflag() {
		return multisiteflag;
	}

	public void setMultisiteflag(boolean multisiteflag) {
		this.multisiteflag = multisiteflag;
	}

}
