package gov.va.cpss.job.sendcbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_OUTPUT_RESOURCE_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.SBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.SBS_OUTPUT_RESOURCE_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobExecutionListener;

import gov.va.cpss.service.SendCBSService;

/**
 * Custom JobExecutionListener used to initialize staging directory and
 * process custom exit status during Send CBS batch job execution.
 * 
 * @author Brad Pickle
 */
public class SendCBSJobExecutionListener implements JobExecutionListener {

    private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());
    
    private SendCBSService sendCBSService;
    
 	public SendCBSService getSendCBSService() {
		return sendCBSService;
	}

	public void setSendCBSService(SendCBSService sendCBSService) {
		this.sendCBSService = sendCBSService;
	}

	@Override
    public void beforeJob(JobExecution jobExecution) {
    	logger.info("Beginning Job Execution");
    	String cbsOutputFileName = jobExecution.getJobParameters().getString(CBS_OUTPUT_FILE_NAME_KEY);
    	jobExecution.getExecutionContext().put(CBS_OUTPUT_RESOURCE_KEY, sendCBSService.getCBSOutputResource(cbsOutputFileName));
    	
    	String sbsOutputFileName = jobExecution.getJobParameters().getString(SBS_OUTPUT_FILE_NAME_KEY);
    	jobExecution.getExecutionContext().put(SBS_OUTPUT_RESOURCE_KEY, sendCBSService.getCBSOutputResource(sbsOutputFileName));
    }

    @Override
    public void afterJob(JobExecution jobExecution) {

        if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
            
            final String exitStatus = jobExecution.getExecutionContext().getString(JOB_FAILURE_KEY);
            
            if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY)) {
                
                final String exitMessage = jobExecution.getExecutionContext().getString(JOB_FAILURE_MESSAGE_KEY);
                
                logger.error("JobFailure detected, setting exit status (" + exitStatus + "): " + exitMessage);
                
                jobExecution.setExitStatus(new ExitStatus(exitStatus, exitMessage));
            } else {
                
            	logger.error("JobFailure detected, setting exit status (" + exitStatus + ")");
                
            	jobExecution.setExitStatus(new ExitStatus(exitStatus));
            }
        } else {
        	jobExecution.setExitStatus(ExitStatus.COMPLETED);
        }

        logger.info("Ending Job Execution");
    }

}
