package gov.va.cpss.job.sendapps;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.job.CbssJobBaseTransactionTasklet;
import gov.va.cpss.service.SendAPPSService;
import gov.va.cpss.service.SftpService;
import gov.va.cpss.time.TimeUtils;

@SuppressWarnings("nls")
public class SendAPPSProcessFileCompletionTasklet extends CbssJobBaseTransactionTasklet {

	private final static String TIMESTAMP_FORMAT = "HH_mm_ss";

	private String tempFilenamePostfix;

	private String serverTargetDirectory;

	private SftpService sftpService;

	private SendAPPSService sendAPPSService;

	public String getTempFilenamePostfix() {
		return this.tempFilenamePostfix;
	}

	public void setTempFilenamePostfix(String inTempFilenamePostfix) {
		this.tempFilenamePostfix = inTempFilenamePostfix;
	}

	public String getServerTargetDirectory() {
		return this.serverTargetDirectory;
	}

	public void setServerTargetDirectory(String inServerTargetDirectory) {
		this.serverTargetDirectory = inServerTargetDirectory;
	}

	public SftpService getSftpService() {
		return this.sftpService;
	}

	public void setSftpService(SftpService inSftpService) {
		this.sftpService = inSftpService;
	}

	public SendAPPSService getSendAPPSService() {
		return this.sendAPPSService;
	}

	public void setSendAPPSService(SendAPPSService inSendAPPSService) {
		this.sendAPPSService = inSendAPPSService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("File Completion Tasklet Running!");

		// Save the jobExecution for reference.
		JobExecution jobExecution = chunkContext.getStepContext().getStepExecution().getJobExecution();

		// Obtain the temporary local file paths as they will be needed
		// regardless of failure.
		// Annual statements file local path
		final String appsOutFilenamePrefix = getJobParameter(jobExecution,
				SendAPPSProcessingConstants.APPS_OUTPUT_FILE_NAME_KEY).replace(".txt", "");
		if ((appsOutFilenamePrefix == null) || appsOutFilenamePrefix.isEmpty()) {
			// Return now and roll back!
			return false;
		}

		// Build list of files in directory for use in next steps
		final List<File> stagedFilesL = new ArrayList<>();
		Arrays.stream(new File(this.sendAPPSService.getStagingDirectory()).listFiles()).forEach(file -> {
			if (file.getName().startsWith(appsOutFilenamePrefix)) {
				stagedFilesL.add(file);
			} else {
				taskletLogger.warn("Skipping file: " + file.getName());
			}
		});

		// If the job has been successful then transfer files to server and
		// attempt to set appropriate final statement and message statuses.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			// Send annual statements file with temporary filename (with
			// timestamp extension).
			// Example:
			// 'APPS-06022016.txt' would be temporarily sent as
			// 'APPS-06022016.txt.<timestamp>.sendapps'
			final Map<File, String> sentFilesTemporaryFilenamesM = new HashMap<>();
			if (successful) {
				for (File file : stagedFilesL) {
					final String tempFilename = getTemporaryFilename(file.getName());

					if (this.sftpService.ftpFileToServerWithName(file.getAbsolutePath(), tempFilename,
							this.serverTargetDirectory)) {
						taskletLogger.info("Sent file to server sucessfully: " + tempFilename);
						sentFilesTemporaryFilenamesM.put(file, tempFilename);
					} else {

						// This is an unexpected and unrecoverable error.
						final String error = "Rollback Triggered - Unsuccessful file transfer: "
								+ getTemporaryFilename(file.getName());
						setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

						// Flag to do rollback.
						successful = false;
					}
				}
			}

			// Mark the APSStmt statuses to sent. (SendAPPSService:L407)
			long totalPatientsIncluded = -1;
			if (successful) {
				final long batchRunProcessId = jobExecution.getJobParameters()
						.getLong(SendAPPSProcessingConstants.APPS_BATCHRUNPROCESS_ID_KEY).longValue();
				totalPatientsIncluded = this.sendAPPSService.updateAllStatementStatusToSent(batchRunProcessId);
				if (totalPatientsIncluded >= 0) {

					taskletLogger.info("Updated all statement status to sent");

				} else {

					// This is an unexpected and unrecoverable error.
					final String error = "Rollback Triggered - Unsuccessful attempting to update all statement status to sent";
					setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

					// Flag to do rollback.
					successful = false;
				}
			}

			// Rename the consolidated statements file on the server (by
			// removing the temporary
			// '.<timestamp>.sendapps' extension).
			long totalFilesIncluded = 0;
			if (successful) {
				for (File file : stagedFilesL) {
					if (this.sftpService.ftpRenameFileInDirectory(sentFilesTemporaryFilenamesM.get(file), file.getName(),
							this.serverTargetDirectory)) {
						taskletLogger.info("Renamed file on server sucessfully: " + file.getName());
						totalFilesIncluded++;
					} else {

						// This is an unexpected and unrecoverable error.
						final String error = "Rollback Triggered - Unsuccessful file rename: " + file.getName();
						setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

						// Flag to do rollback.
						successful = false;
					}
				}
			}

			// Send a done file for each file to the server.
			if (successful) {
				for (File file : stagedFilesL) {
					final String appsOutGetDoneFilename = getDoneFilename(file.getName());

					if (this.sftpService.ftpEmptyFileToServerWithName(appsOutGetDoneFilename, this.serverTargetDirectory)) {
						taskletLogger.info("Transfer done file to server sucessfully: " + appsOutGetDoneFilename);
					} else {

						// This is an unexpected and unrecoverable error.
						final String error = "Rollback Triggered - Unsuccessful done file transfer: "
								+ appsOutGetDoneFilename;
						setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

						// Flag to do rollback.
						successful = false;
					}
				}
			}

			// If successful then save final statistics.
			if (successful) {
				// Save batch job completion results.
				jobExecution.getExecutionContext().putLong(SendAPPSProcessingConstants.APPS_FILES_SENT_KEY,
						totalFilesIncluded);
				jobExecution.getExecutionContext().putLong(SendAPPSProcessingConstants.APPS_STATEMENTS_SENT_KEY,
						totalPatientsIncluded);
			}
		}

		for (File file : stagedFilesL) {
			// Regardless of failure or successful batch run, delete local
			// statement files if exist.
			if (this.sendAPPSService.deleteAPPSOutputResource(file.getAbsolutePath())) {
				taskletLogger.info("Deleted temporary local file sucessfully: " + file.getAbsolutePath());
			} else {

				// Don't make this a failure in case transfer had already been
				// successful.
				taskletLogger.warn("Unable to delete the temporary local file: " + file.getAbsolutePath());
			}
		}

		return successful;
	}

	private String getJobParameter(final JobExecution jobExecution, final String jobParameterKey) {

		final String jobParameter = jobExecution.getJobParameters().getString(jobParameterKey);

		// If an error getting the jobParameter then set the failure status and
		// trigger roll back.
		if ((jobParameter != null) && !jobParameter.isEmpty()) {

			taskletLogger.info(jobParameterKey + ": " + jobParameter);

		} else {

			// This is an unexpected and unrecoverable error.
			final String error = "Rollback Triggered - Could not obtain the job parameter, " + jobParameterKey;
			setFailureStatus(jobExecution, DATA_ERROR_STATUS, error);

		}

		return jobParameter;
	}

	/**
	 * Build a temporary filename for use on the target sftp server. The format
	 * of the temporary filename: Given filename: 'CBS-06022016.txt' would
	 * return temporary filename: 'CBS-06022016.txt.<timestamp>.sendapps'
	 * 
	 * @param filename
	 *            The base filename.
	 * @return The temporary filename.
	 */
	private String getTemporaryFilename(final String filename) {
		if ((filename != null) && !filename.isEmpty()) {
			StringBuilder tempFilename = new StringBuilder(filename);
			tempFilename.append(".");
			tempFilename.append(TimeUtils.getTimestamp(TIMESTAMP_FORMAT));
			tempFilename.append(".");
			tempFilename.append(this.tempFilenamePostfix);
			return tempFilename.toString();
		}
		return null;
	}

	/**
	 * Get the done filename based on the output filename. The done filename is
	 * the same base filename but with ".don" extension instead of ".txt"
	 * 
	 * @param filename
	 *            The output filename.
	 * @return The done filename.
	 */
	@SuppressWarnings("static-method")
	private String getDoneFilename(final String filename) {
		return filename.substring(0, filename.lastIndexOf(".txt")) + ".don";
	}

}
