package gov.va.cpss.job.sendapps;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.BeforeStep;

import gov.va.cpss.job.CBSSPagingMultiItemWriter;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.sendapps.APSStmtStorageBean;
import gov.va.cpss.model.sendapps.SendAPPSRecord;
import gov.va.cpss.service.SendAPPSService;

@SuppressWarnings("nls")
public class SendAPPSItemWriter extends CBSSPagingMultiItemWriter<APSStmt>{

	private SendAPPSService sendAPPSService;

	public SendAPPSItemWriter() {
	}
	
	public SendAPPSService getSendAPPSService() {
		return this.sendAPPSService;
	}
	
	public void setSendAPPSService(SendAPPSService inSendAPPSService) {
		this.sendAPPSService = inSendAPPSService;
	}
	
	@Override
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		super.beforeStep(stepExecution);

		// Store the APSStmt list in a storage bean to be used in the write and send step later
		APSStmtStorageBean.setApsStmtsList(new ArrayList<APSStmt>());
	}

	@SuppressWarnings("resource") // For BufferedWriter writer
	@Override
	protected boolean doWrite(List<? extends APSStmt> inItems) {
		try {
			final String apsFilePath = this.sendAPPSService.getStagingDirectory() + File.separator
					+ this.jobExecution.getJobParameters().getString(SendAPPSProcessingConstants.APPS_OUTPUT_FILE_NAME_KEY);
			
			this.logger.info("Writing APSStmt to file: " + apsFilePath); 
			
			final List<APSStmt> writeItems = new ArrayList<>();
			if(Objects.nonNull(inItems)) {
				inItems.forEach(apsStmt -> writeItems.add(apsStmt));
			} else {
				this.logger.error("Could not load APSStmts for write.");
				return false;
			}
			
			final long totalPatients = this.jobExecution.getExecutionContext().getLong(SendAPPSProcessingConstants.APPS_TOTAL_PATIENTS_INCLUDED_KEY, 0);
			this.jobExecution.getExecutionContext().putLong(SendAPPSProcessingConstants.APPS_TOTAL_PATIENTS_INCLUDED_KEY, totalPatients + writeItems.size());
			this.logger.info("Total patients included: " + this.jobExecution.getExecutionContext().getLong(SendAPPSProcessingConstants.APPS_TOTAL_PATIENTS_INCLUDED_KEY));
			
			final SendAPPSWriteAPPSRecordSource source = new SendAPPSWriteAPPSRecordSource();
			source.setApsStmtList(writeItems);
			
			final BufferedWriter writer;
			if(new File(apsFilePath).exists()) {
				writer = Files.newBufferedWriter(Paths.get(apsFilePath), StandardOpenOption.APPEND);
			} else {
				writer = Files.newBufferedWriter(Paths.get(apsFilePath));
			}
			
			while(source.apsRecordsRemaining()) {
				final SendAPPSRecord<?> currentRecord = source.nextAPSRecord();
				if(currentRecord == null) {
					break;
				}
				
				writer.write(currentRecord.toFileOutputString());
				writer.newLine();
			}
			
			writer.close();
		} catch (IOException e) {
			this.logger.error(e);
			return false;
		}
		
		return true;
	}
}
