package gov.va.cpss.job.printack;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_RECEIVED_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_UPDATED_KEY;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.job.UpdatePrintAckJob;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PAFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;
import gov.va.cpss.service.UpdatePrintAckService;

/**
 * Custom ItemWriter used by the UpdatePrintAck job to process the print
 * acknowledgements and update the consolidated statement statuses accordingly.
 * 
 * @author Brad Pickle
 *
 */
public class UpdatePrintAckItemWriter implements ItemWriter<PAFileRecord>, ItemWriteListener<PAFileRecord> {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt writes.
	 */
	private boolean forceStop = false;

	private JobExecution jobExecution;

	private UpdatePrintAckService updatePrintAckService;

	public UpdatePrintAckService getUpdatePrintAckService() {
		return updatePrintAckService;
	}

	public void setUpdatePrintAckService(UpdatePrintAckService updatePrintAckService) {
		this.updatePrintAckService = updatePrintAckService;
	}

	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		logger.info("Before Step Execution");

		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public void write(List<? extends PAFileRecord> paFileRecordL) throws Exception {
		// Do not attempt to write if forced stop.
		if (forceStop) {
			return;
		}

		try {
			List<Long> printAckRecIdList = UpdatePrintAckJob.getCurrentFilePrintAckRecIdList(jobExecution);
			List<Long> currentPrintAckRecIdList = new ArrayList<Long>();

			for (PAFileRecord paFileRecord : paFileRecordL) {
				logger.debug("Write:" + paFileRecord);

				final PrintAckRec printAckRec = paFileRecord.getPrintAckRec();

				// Create a PrintAckRec record for this PA
				currentPrintAckRecIdList.add(updatePrintAckService.initializePrintAck(printAckRec));

				logger.debug("StatementDate:" + paFileRecord.getStatementDate());
				logger.debug("PrintDate:" + paFileRecord.getPrintDate());

				final List<ADFileRecord> missingPatientAccts = new ArrayList<ADFileRecord>();

				// Update the printAckId, printDate and status=ACK for
				// statements corresponding to the patient accounts in this PA's
				// list of AD records
				final int numStmtsUpdated = updatePrintAckService.processPrintAcks(printAckRec,
						paFileRecord.getStatementDate(), paFileRecord.getPrintDate(), missingPatientAccts);
				logger.debug("numStmtsUpdated:" + numStmtsUpdated);

				if (stopJobIfMissingStatements(missingPatientAccts, paFileRecord)) {
					// Stop processing file immediately if there are AD record
					// patient accounts with no statements.
					break;
				}

				// Update the status to SUCCESS for this PA
				updatePrintAckService.setPrintAckSuccess(printAckRec);

				// Received and updated will always be the same since no longer
				// allowing missing patient statements. May need to remove
				// RECEIVED at some point.
				accrueProcessingCount(TOTAL_CBS_RECEIVED_KEY, numStmtsUpdated);
				accrueProcessingCount(TOTAL_CBS_UPDATED_KEY, numStmtsUpdated);
			}

//			if (!forceStop) {
				// Delay accruing these ids till now, since exceptions in the
				// previous code may roll back.
				printAckRecIdList.addAll(currentPrintAckRecIdList);
				UpdatePrintAckJob.putCurrentFilePrintAckRecIdList(jobExecution, printAckRecIdList);
//			}
		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Error during write because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}

	}

	/**
	 * Accrue the given processing count in the job execution context.
	 * 
	 * @param key
	 *            Name of the processing count.
	 * @param numStmts
	 *            Amount to add to the processing count.
	 */
	private void accrueProcessingCount(final String key, final int numStmts) {
		long totalStmts = jobExecution.getExecutionContext().getLong(key);
		totalStmts += numStmts;
		jobExecution.getExecutionContext().putLong(key, totalStmts);
	}

	/**
	 * Check the given List of ADFileRecords representing AD records where no
	 * corresponding patient statement exists for the statement date. If the
	 * List is not empty, then stop the job.
	 * 
	 * @param missingPatientAccts
	 *            List of ADFileRecords for patients where no corresponding
	 *            patient statement exists.
	 * @param paFileRecord
	 *            The PAFileRecord with the missing patient accounts
	 * @return true if there are missing patient statements and the job should
	 *         stop.
	 */
	private boolean stopJobIfMissingStatements(List<ADFileRecord> missingPatientAccts, PAFileRecord paFileRecord) {
		final boolean isMissingStatements = (missingPatientAccts.size() > 0);

		if (isMissingStatements) {
			UpdatePrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, missingPatientAccts);
			
			StringBuilder error = new StringBuilder();
			error.append("Invalid patient accounts encountered for facility, ");
			error.append(paFileRecord.getFacilityNum());
			error.append(". No patient statement found for the following patient accounts: ");

			String delim = "";
			for (ADFileRecord adFileRecord : missingPatientAccts) {
				error.append(delim);
				error.append(adFileRecord.getPatientAccount());
				delim = ", ";
			}

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}
		return isMissingStatements;
	}

	@Override
	public void beforeWrite(List<? extends PAFileRecord> items) {
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			logger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterWrite(List<? extends PAFileRecord> items) {
	}

	@Override
	public void onWriteError(Exception e, List<? extends PAFileRecord> items) {

		logger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to write item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, error.toString());
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		logger.error("Writer execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		logger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		logger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
