package gov.va.cpss.job.printack;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.INCOMPLETE_FILE_ERROR_STATUS;

import static gov.va.cpss.job.printack.UpaProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.DATE_RECEIVED_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_RECEIVED_KEY;
import static gov.va.cpss.job.printack.UpaProcessingConstants.TOTAL_CBS_UPDATED_KEY;

import java.sql.Timestamp;
import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.job.UpdatePrintAckJob;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PAFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Custom ItemProcessor used by the UpdatePrintAck job to validate the
 * PAFileRecord instantiated from each line of the input file and build a
 * PrintAckRec from it.
 * 
 * @author Brad Pickle
 */
public class UpdatePrintAckItemProcessor implements ItemProcessor<PAFileRecord, PAFileRecord> {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());;

	private JobExecution jobExecution;

	private int batchRunId;
	private String fileName;
	private Timestamp dateReceived;

	private long lineNumber = 0;
	private PAFileRecord previousPAFileRecord = null;

	/**
	 * Before processing capture the job execution and job parameters for the
	 * job and initialize processing counts.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		logger.info("Before Step Execution");

		// Save the job execution at the beginning of the step.
		jobExecution = stepExecution.getJobExecution();

		// Get the id of the BatchRun record for this job.
		batchRunId = jobExecution.getJobParameters().getLong(BATCH_RUN_ID_KEY).intValue();

		// Get the name of the file being processed.
		fileName = jobExecution.getJobParameters().getString(INPUT_RESOURCE_KEY);

		// Get the date the file being processed was received.
		dateReceived = new Timestamp(jobExecution.getJobParameters().getLong(DATE_RECEIVED_KEY));

		// Initialize the processing counts.
		jobExecution.getExecutionContext().putLong(TOTAL_CBS_RECEIVED_KEY, 0);
		jobExecution.getExecutionContext().putLong(TOTAL_CBS_UPDATED_KEY, 0);
		UpdatePrintAckJob.putCurrentFilePrintAckRecIdList(jobExecution, new ArrayList<Long>());
		UpdatePrintAckJob.putCurrentFileMissingPatientAcctList(jobExecution, new ArrayList<ADFileRecord>());
		
		lineNumber = 0;
		previousPAFileRecord = null;
	}

	@Override
	public PAFileRecord process(PAFileRecord paFileRecord) throws Exception {
		boolean success = true;
		lineNumber++;
		PrintAckRec printAckRec = null;
		PAFileRecord returnPAFileRecord = null;

		if (success) {
			success = validatePAFileRecord(paFileRecord);
		}

		if (success) {
			if ((previousPAFileRecord != null)
					&& (!paFileRecord.getFacilityNum().equals(previousPAFileRecord.getFacilityNum()))) {
				success = lastSiteCompleted();
				if (!success) {
					stopJob(INCOMPLETE_FILE_ERROR_STATUS);
				}
			}
		}

		// Save the PAFileRecord so that seqNum and totSeqNum can be validated
		// next process
		previousPAFileRecord = paFileRecord;

		if (success) {

			printAckRec = new PrintAckRec();

			printAckRec.setBatchRunId(batchRunId);
			printAckRec.setFileName(fileName);
			printAckRec.setDateReceived(dateReceived);

			printAckRec.setStationNum(paFileRecord.getFacilityNum());
			printAckRec.setPrintAckDetailL(paFileRecord.getPrintAckDetailL());
			
			returnPAFileRecord = paFileRecord;
			paFileRecord.setPrintAckRec(printAckRec);
		}

		return returnPAFileRecord;
	}

	/**
	 * After processing check for unexpected conditions that suggest an error.
	 * 
	 * @param stepExecution
	 */
	@AfterStep
	public void afterStep(StepExecution stepExecution) {

		// If no other error detected then check for other possible error
		// conditions.
		if (!jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {

			// If read count is zero then report a job failure.
			if (stepExecution.getReadCount() == 0) {
				setFailureStatusAndMessage(EMPTY_FILE_ERROR_STATUS, "Input file is empty");
			} else if (!lastSiteCompleted()) {
				setFailureStatus(INCOMPLETE_FILE_ERROR_STATUS);
			}

		}

		logger.info("After Step Execution");
	}

	/**
	 * Validate the PAFileRecord instantiated from the current line of the input
	 * file. Job will be stopped if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord is valid.
	 */
	private boolean validatePAFileRecord(PAFileRecord paFileRecord) {

		if (paFileRecord == null) {
			stopJob(PROCESSING_FAILURE_STATUS, "Attempted to process unknown row at line " + lineNumber + ".");
			return false;
		}
		
		logger.debug("Validadating:" + paFileRecord);

		if (!validateType(paFileRecord)) {
			return false;
		}

		if (!validateFacilityNum(paFileRecord)) {
			return false;
		}

		if (!validateSeqNum(paFileRecord)) {
			return false;
		}

		if (!validateTotSeqNum(paFileRecord)) {
			return false;
		}

		if (!validateTotStatement(paFileRecord)) {
			return false;
		}

		return true;
	}

	/**
	 * Validate the type property of the given PAFileRecord. Job will be stopped
	 * if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord type is valid.
	 */
	private boolean validateType(PAFileRecord paFileRecord) {

		if (paFileRecord.getType() != PAFileRecord.DataType.PA) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Unknown record type at line " + lineNumber + ". Expected " + PAFileRecord.DataType.PA + ".");
			return false;
		}

		return true;
	}

	/**
	 * Validate the facilityNum property of the given PAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord facilityNum is valid.
	 */
	private boolean validateFacilityNum(PAFileRecord paFileRecord) {
		String facilityNum = paFileRecord.getFacilityNum();
		facilityNum = (facilityNum == null) ? "" : facilityNum.trim();
		paFileRecord.setFacilityNum(facilityNum);

		if (facilityNum.length() == 0) {
			stopJob(PROCESSING_FAILURE_STATUS, "Empty FACILITY-NUM at line " + lineNumber + ".");
			return false;
		}
		return true;
	}

	/**
	 * Validate the seqNum property of the given PAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord seqNum is valid.
	 */
	private boolean validateSeqNum(PAFileRecord paFileRecord) {

		if (paFileRecord.getSeqNum() < 1) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM=" + paFileRecord.getSeqNum());
			return false;
		}

		if ((previousPAFileRecord != null)
				&& paFileRecord.getFacilityNum().equals(previousPAFileRecord.getFacilityNum())) {
			if (paFileRecord.getSeqNum() != (previousPAFileRecord.getSeqNum() + 1)) {
				stopJob(PROCESSING_FAILURE_STATUS, "Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM="
						+ paFileRecord.getSeqNum() + "  Expected " + (previousPAFileRecord.getSeqNum() + 1) + ".");
				return false;
			}
		} else {
			// First record in file, or a new facility num
			if (paFileRecord.getSeqNum() != 1) {
				stopJob(PROCESSING_FAILURE_STATUS, "Invalid SEQ-NUM at line " + lineNumber + ". SEQ-NUM="
						+ paFileRecord.getSeqNum() + "  Expected 1.");
				return false;
			}
		}

		return true;
	}

	/**
	 * Validate the totSeqNum property of the given PAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord totSeqNum is valid.
	 */
	private boolean validateTotSeqNum(PAFileRecord paFileRecord) {

		if (paFileRecord.getTotSeqNum() < 0) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid TOT-SEQ-NUM at line " + lineNumber + ". TOT-SEQ-NUM=" + paFileRecord.getTotSeqNum());
			return false;
		}

		if ((paFileRecord.getTotSeqNum() != 0) && (paFileRecord.getTotSeqNum() < paFileRecord.getSeqNum())) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Invalid TOT-SEQ-NUM or SEQ-NUM at line " + lineNumber + ". TOT-SEQ-NUM="
							+ paFileRecord.getTotSeqNum() + ", SEQ-NUM=" + paFileRecord.getSeqNum()
							+ ".  TOT-SEQ-NUM must be greater than or equal to SEQ-NUM.");
			return false;
		}

		return true;
	}

	/**
	 * Validate the totStatement property of the given PAFileRecord. Job will be
	 * stopped if invalid.
	 * 
	 * @param paFileRecord
	 *            PAFileRecord to validate.
	 * @return boolean indicating if PAFileRecord totStatement is valid.
	 */
	private boolean validateTotStatement(PAFileRecord paFileRecord) {

		if (paFileRecord.getTotStatement() != paFileRecord.getPrintAckDetailL().size()) {
			stopJob(PROCESSING_FAILURE_STATUS,
					"Number of AD records does not match TOT-STATEMENT at line " + lineNumber + ".  AD records found:"
							+ paFileRecord.getPrintAckDetailL().size() + "  TOT-STATEMENT="
							+ paFileRecord.getTotStatement());
			return false;
		}

		return true;
	}

	/**
	 * Verify the last site processing was completed.
	 * 
	 * @return Boolean indicating if processing completed.
	 */
	private boolean lastSiteCompleted() {

		boolean successful = true;

		if ((previousPAFileRecord != null)
				&& (previousPAFileRecord.getTotSeqNum() != previousPAFileRecord.getSeqNum())) {
			successful = false;
			setFailureMessage("Incomplete sequence for FACILITY-NUM=" + previousPAFileRecord.getFacilityNum()
					+ ". Expected TOT-SEQ-NUM at line " + lineNumber + " to be " + previousPAFileRecord.getSeqNum()
					+ " but it was " + previousPAFileRecord.getTotSeqNum() + ".");
		}

		return successful;
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status) {
		// Log message.
		logger.error("Processor execution encountered unrecoverable error and forced stop");
		// Set failure and message.
		setFailureStatus(status);
		// Stop job.
		jobExecution.stop();
	}

	/**
	 * Forcefully stop the job processing because an error was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status, final String message) {
		// Set failure.
		stopJob(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	private void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);
		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		logger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		logger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
