package gov.va.cpss.job.fps;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.fps.FpsProcessingConstants.PATIENT_ID_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.SITE_ID_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_SITE_COUNT_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.TOTAL_STATEMENT_COUNT_KEY;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.service.FpsService;

/**
 * Implementation of ItemWriter used to handle writing of raw records when
 * processing.
 * 
 * @author DNS   
 */
public class FpsDelegateMultiItemWriter
		implements ItemWriter<PSRecord>, ItemWriteListener<PSRecord>, StepExecutionListener {

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt writes.
	 */
	private boolean forceStop = false;

	private final Logger writerLogger = Logger.getLogger(FpsDelegateMultiItemWriter.class.getCanonicalName());

	private JobExecution jobExecution;

	private FpsService fpsService;

	private Long siteId;
	private Long patientId;

	/*
	 * Delegate writer used to take advantage of batch writing patient details
	 * records.
	 */
	private ItemWriter<PSDetails> psDetailsDatabaseItemWriter;

	public FpsService getFpsService() {
		return fpsService;
	}

	public void setFpsService(FpsService fpsService) {
		this.fpsService = fpsService;
	}

	public ItemWriter<PSDetails> getPsDetailsDatabaseItemWriter() {
		return psDetailsDatabaseItemWriter;
	}

	public void setPsDetailsDatabaseItemWriter(ItemWriter<PSDetails> psDetailsDatabaseItemWriter) {
		this.psDetailsDatabaseItemWriter = psDetailsDatabaseItemWriter;
	}

	@Override
	public void write(List<? extends PSRecord> items) throws Exception {

		writerLogger.debug("Begin Write");

		// Do not attempt to write if forced stop.
		if (forceStop) {
			return;
		}

		try {

			// List of details that can be written using batch write.
			List<PSDetails> detailsL = new ArrayList<>();

			// Loop through all of the statements to write.
			for (Object item : items) {

				// Update appropriate objects with the appropriate FK values.
				// The assumption is the rows are ordered properly.

				// The data is processed as:
				// PS, then subsequent 1-* PH.
				// PH, then subsequent 1-* PD.

				if (item instanceof PSSite) {

					PSSite site = (PSSite) item;

					// If a PSSite row then reset the child keys to null to
					// prevent
					// erroneous foreign key references.
					resetSiteFK();

					// Batch won't return generated keys so use DAO.
					writerLogger.debug("Saving PSSite: " + site.toString());
					fpsService.saveSite(site);

					// Save this record ID for later reference for children
					// PSPatient records.
					updateSiteProcessingData(site.getId());

				} else if (item instanceof PSPatient) {

					PSPatient patient = (PSPatient) item;

					// The SiteId FK needs to be set in the associated PSPatient
					// object.
					// Save the PSSite ID.
					if (jobExecution.getExecutionContext().containsKey(SITE_ID_KEY)) {
						siteId = jobExecution.getExecutionContext().getLong(SITE_ID_KEY);
					}

					// If a PSPatient row then reset the child keys to null to
					// prevent
					// erroneous foreign key references.
					resetPatientFK();

					if (siteId != null) {

						// For PSPatient records, set the database
						// auto-generated
						// Site ID in the patient record.
						patient.setPsSite(new PSSite(siteId));

						// Batch won't return generated keys so use service.
						writerLogger.debug("Saving PSPatient: " + patient.toString());
						fpsService.savePatient(patient);

						// Save this record ID for later reference for children
						// PSDetails records.
						updatePatientProcessingData(patient.getId());
					} else {

						// Unrecoverable error so stop the job.
						stopJob(WRITE_FAILURE_STATUS, "Attempted to process PH row with a null PS FK");
						break;
					}

				} else if (item instanceof PSDetails) {

					PSDetails details = (PSDetails) item;

					// The PatientId FK needs to be set in the associated
					// PSDetails
					// object.
					// Save the PSPatient ID.
					if (jobExecution.getExecutionContext().containsKey(PATIENT_ID_KEY)) {
						patientId = jobExecution.getExecutionContext().getLong(PATIENT_ID_KEY);
					}

					// At this point patientId FK should not be null.
					if (patientId != null) {

						// For PSDetails records, set the database
						// auto-generated
						// Patient ID in the details record.
						details.setPsPatientId(patientId);

						// Because we don't need generated keys so we can use
						// the
						// batch for
						// this insert. They will be written to the database
						// after
						// the record loop has finished.
						detailsL.add(details);
					} else {

						// Unrecoverable error so stop the job.
						stopJob(WRITE_FAILURE_STATUS, "Attempted to process PD row with a null PH FK");
						break;
					}

				} else {

					// Unrecoverable error so stop the job.
					stopJob(WRITE_FAILURE_STATUS, "Attempted to write unknown record");
					break;
				}
			}

			// We can use batch to update replaced statements.
			if (!forceStop && !detailsL.isEmpty()) {
				writerLogger.debug("Saving list of CBSSiteTrans");
				psDetailsDatabaseItemWriter.write(detailsL);
			}

		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Error during write because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());
			if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
				error.append("\nCause: ");
				error.append(e.getCause().getMessage().trim());
			}

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}

		writerLogger.debug("End Write");
	}

	/**
	 * Reset the site values from the job execution to prepare to process next
	 * site.
	 */
	private void resetSiteFK() {
		siteId = null;
		if (jobExecution.getExecutionContext().containsKey(SITE_ID_KEY)) {
			jobExecution.getExecutionContext().remove(SITE_ID_KEY);
		}

		resetPatientFK();
	}

	/**
	 * Reset the patient values from the job execution to prepare to process
	 * next patient.
	 */
	private void resetPatientFK() {
		patientId = null;
		if (jobExecution.getExecutionContext().containsKey(PATIENT_ID_KEY)) {
			jobExecution.getExecutionContext().remove(PATIENT_ID_KEY);
		}
	}

	/**
	 * Update site processing metadata.
	 */
	private boolean updateSiteProcessingData(final Long primaryKeyResult) {

		boolean successful = true;

		writerLogger.debug("Wrote PS(" + primaryKeyResult + ")");
		jobExecution.getExecutionContext().putLong(SITE_ID_KEY, primaryKeyResult);

		if (jobExecution.getExecutionContext().containsKey(TOTAL_SITE_COUNT_KEY)) {
			jobExecution.getExecutionContext().putLong(TOTAL_SITE_COUNT_KEY,
					jobExecution.getExecutionContext().getLong(TOTAL_SITE_COUNT_KEY) + 1);
		} else {
			setFailureMessage("Unable to increment total site count");
			successful = false;
		}

		return successful;
	}

	/**
	 * Update patient processing metadata.
	 */
	private boolean updatePatientProcessingData(final Long primaryKeyResult) {

		boolean successful = true;

		writerLogger.debug("Wrote PH(" + primaryKeyResult + ")");
		jobExecution.getExecutionContext().putLong(PATIENT_ID_KEY, primaryKeyResult);

		if (jobExecution.getExecutionContext().containsKey(TOTAL_STATEMENT_COUNT_KEY)) {
			jobExecution.getExecutionContext().putLong(TOTAL_STATEMENT_COUNT_KEY,
					jobExecution.getExecutionContext().getLong(TOTAL_STATEMENT_COUNT_KEY) + 1);
		} else {
			setFailureMessage("Unable to increment total statement count");
			successful = false;
		}

		return successful;
	}

	@Override
	public void beforeWrite(List<? extends PSRecord> items) {
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			writerLogger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterWrite(List<? extends PSRecord> items) {
		// Do not need to do anything.
	}

	@Override
	public void onWriteError(Exception e, List<? extends PSRecord> items) {

		writerLogger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to write item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, "Unrecoverable writer error");
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		writerLogger.info("Before Step Execution");
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		writerLogger.info("After Step Execution");
		// Do not need to do anything after step.
		return null;
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		writerLogger.error("Writer execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		writerLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		writerLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
