package gov.va.cpss.job.cbs;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.log4j.Logger;

import gov.va.cpss.model.fps.PSPatient;

/**
 * Implementation of the interface for the runtime parameters used by the
 * Generate CBS batch job during processing.
 * 
 * @author DNS   
 */
public class CbsRuntimeStateImpl implements CbsRuntimeState {

	private static final Logger parameterLogger = Logger.getLogger(CbsRuntimeStateImpl.class.getCanonicalName());

	/**
	 * Boolean flag that indicates there is no data to process.
	 */
	private boolean empty = false;

	/**
	 * Boolean flag that indicates if an error reading NEW data.
	 */
	private boolean dataError = false;

	/**
	 * Queue of PSReceived ID relevant to the job.
	 */
	private Queue<Long> psReceivedIdQ = new LinkedList<>();

	/**
	 * List of processed PSReceived ID.
	 */
	private List<Long> psReceivedIdProcessL = new ArrayList<>();

	/**
	 * The currently processing PSReceived ID.
	 */
	private Long currentPSReceivedID = null;

	/**
	 * Queue of PSSite ID relevant to the job.
	 */
	private Queue<Long> psSiteIdQ = new LinkedList<>();

	/**
	 * The currently processing PSSite.
	 */
	private Long currentPSSiteID = null;

	/**
	 * The Map of CBS Account ID to a list of all new PSPatient records.
	 */
	private Map<Long, List<PSPatient>> psPatientM = new LinkedHashMap<>();
	
	/**
	 * The current entry in the Map of CBS Account ID to list of all new PSPatient records.
	 */
	private Entry<Long, List<PSPatient>> currentPSPatientEntry = null;

	@Override
	public void initialize() {
		empty = false;
		dataError = false;
		psReceivedIdQ.clear();
		psReceivedIdProcessL.clear();
		psSiteIdQ.clear();
		currentPSReceivedID = null;
		currentPSSiteID = null;
	}

	@Override
	public boolean isEmpty() {
		return empty;
	}

	@Override
	public void setEmpty(boolean empty) {
		this.empty = empty;
	}

	@Override
	public boolean isDataError() {
		return dataError;
	}

	@Override
	public void setDataError(boolean dataError) {
		this.dataError = dataError;
	}

	@Override
	public boolean addPSReceivedEntry(final Long id) {
		return psReceivedIdQ.add(id);
	}

	@Override
	public boolean addPSSiteEntry(final Long psSite) {
		return psSiteIdQ.add(psSite);
	}

	@Override
	public List<Long> getPSReceivedIDList() {
		return psReceivedIdProcessL;
	}

	@Override
	public Long getCurrentPSReceivedID() {
		return currentPSReceivedID;
	}

	@Override
	public Long getCurrentPSSiteID() {
		return currentPSSiteID;
	}

	@Override
	public Long pollCurrentPSReceivedID() {
		// Poll the next PSReceived ID.
		currentPSReceivedID = psReceivedIdQ.poll();
		// Save the polled ID to the processed list.
		if (currentPSReceivedID != null) {
			psReceivedIdProcessL.add(currentPSReceivedID);
		}
		return getCurrentPSReceivedID();
	}

	@Override
	public Long pollCurrentPSSiteID() {
		currentPSSiteID = psSiteIdQ.poll();
		return getCurrentPSSiteID();
	}

	@Override
	public boolean savePSPatientMap(Map<Long, List<PSPatient>> queryM) {
		boolean successful = false;

		if (queryM.isEmpty()) {

			parameterLogger.warn("Attempted to save empty PSPatient map");
		} else {

			parameterLogger.debug("savePSPatientMap, queryM size = " + queryM.size());

			parameterLogger.debug("savePSPatientMap, attempting merging...");

			// Merge the newly queried map into the composite map.
			psPatientM = Stream.of(psPatientM, queryM).map(Map::entrySet).flatMap(Set::stream)
					.collect(Collectors.toMap(Entry::getKey, Entry::getValue, (a, b) -> {
						List<PSPatient> both = new ArrayList<>(a);
						both.addAll(b);
						return both;
					}));

			parameterLogger.debug("savePSPatientMap, psPatientM size = " + psPatientM.size());

			successful = true;
		}

		return successful;
	}
	
	@Override
	public Map.Entry<Long, List<PSPatient>> getCurrentPSPatientM() {
		return currentPSPatientEntry;
	}

	@Override
	public Map.Entry<Long, List<PSPatient>> pollCurrentPSPatientM() {

		Iterator<Map.Entry<Long, List<PSPatient>>> i = psPatientM.entrySet().iterator();
		if (i.hasNext()) {
			currentPSPatientEntry = i.next();
			i.remove();
		} else {
			currentPSPatientEntry = null;
		}

		return getCurrentPSPatientM();
	}

}
