package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY;

import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.service.CbsService;

/**
 * Complete processing by reverting any REPLACED statements to NEW state and
 * removing the processed consolidated statements in INITIAL state.
 * 
 * @author DNS   
 */
public class CbsProcessStmtSystemFailureCompletionTasklet extends CbsJobBaseCompletionTasklet {

	private CbsService cbsService;

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("Stmt System Failure Tasklet Running!");

		// Revert REPLACED statements to NEW that are associated with INITIAL Statements
		final int replacedCount = cbsService.revertReplacedStatementsAssociatedWithInitialStatements();

		// Verify the number reverted matched the running total from the writer.
		final long expectedReplacedStatementCount = chunkContext.getStepContext().getStepExecution().getJobExecution()
				.getExecutionContext().getLong(CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY);

		if (replacedCount == expectedReplacedStatementCount) {
			taskletLogger.info("Count of statements reverted to NEW (" + replacedCount + ")");
		} else {
			// If the counts do not match then this is an unexpected and
			// unrecoverable error.
			final String error = "Rollback Triggered - Updated (" + replacedCount
					+ ") REPLACED status rows but expected to update (" + expectedReplacedStatementCount
					+ ") rows for CBSStmt";
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// Delete all INITIAL statements.
		final int deletedCount = cbsService.removeInitialStatements();

		// Verify the number published matched the running total from the
		// writer.
		final long expectedDeletedCount = chunkContext.getStepContext().getStepExecution().getJobExecution()
				.getExecutionContext().getLong(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY);

		if (deletedCount == expectedDeletedCount) {
			taskletLogger.info("Count of initial statements deleted (" + deletedCount + ")");
		} else {
			// If the counts do not match then this is an unexpected and
			// unrecoverable error.
			final String error = "Rollback Triggered - Deleted (" + deletedCount
					+ ") INITIAL status rows but expected to delete (" + expectedDeletedCount + ") rows for CBSStmt";
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// Set the completion statistics.
		updateStatistics(chunkContext.getStepContext().getStepExecution().getJobExecution(), 0);

		return successful;
	}

}
