package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY;

import java.util.List;

import org.springframework.batch.core.scope.context.ChunkContext;

import gov.va.cpss.service.CbsService;
import gov.va.cpss.service.FpsService;

/**
 * Complete processing by setting the processed consolidated statements to NEW
 * state and setting the processed PSReceived to PROCESSED state.
 * 
 * @author DNS  
 */
public class CbsProcessStmtDataCompletionTasklet extends CbsJobBaseCompletionTasklet {

	private CbsRuntimeState cbsRuntimeState;

	private FpsService fpsService;

	private CbsService cbsService;

	public CbsRuntimeState getCbsRuntimeState() {
		return cbsRuntimeState;
	}

	public void setCbsRuntimeState(CbsRuntimeState cbsRuntimeState) {
		this.cbsRuntimeState = cbsRuntimeState;
	}

	public FpsService getFpsService() {
		return fpsService;
	}

	public void setFpsService(FpsService fpsService) {
		this.fpsService = fpsService;
	}

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	@Override
	protected boolean executeLogic(ChunkContext chunkContext) {

		boolean successful = true;

		taskletLogger.info("Stmt Completion Tasklet Running!");

		// Publish all INITIAL statements to NEW status.
		long newStatementCount = cbsService.publishAllNewStatements();

		// Verify the number published matched the running total from the
		// writer.
		final long expectedNewStatementCount = chunkContext.getStepContext().getStepExecution().getJobExecution()
				.getExecutionContext().getLong(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY);

		if (newStatementCount == expectedNewStatementCount) {
			taskletLogger.info("CBSStmt Records New Count: " + newStatementCount);
		} else {
			// If the counts do not match then this is an unexpected and
			// unrecoverable error.
			final String error = "Rollback Triggered - Updated (" + newStatementCount
					+ ") INITIAL to NEW rows but expected to update (" + expectedNewStatementCount
					+ ") rows for CBSStmt";
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// Publish the processed PSReceived records to PROCESSED status.
		final List<Long> psReceivedIDL = cbsRuntimeState.getPSReceivedIDList();

		// Verify the number published matched the running total from the
		// writer.
		final int receivedUpdateCount = fpsService.processPSReceivedList(psReceivedIDL);

		if (receivedUpdateCount == psReceivedIDL.size()) {
			taskletLogger.info("PSReceived Records Processed Count: " + receivedUpdateCount);
		} else {
			// If the counts do not match then this is an unexpected and
			// unrecoverable error.
			final String error = "Rollback Triggered - Updated (" + receivedUpdateCount
					+ ") NEW to PROCESSED rows but expected to update (" + psReceivedIDL.size()
					+ ") rows for PSReceived";
			setFailureStatus(chunkContext.getStepContext().getStepExecution().getJobExecution(), DATA_ERROR_STATUS,
					error);

			// Flag to do rollback.
			successful = false;
		}

		// If a database error occurred a rollback will be initiated so set new
		// statement count to zero.
		// Statements will remain in INITIAL state.
		if (!successful) {
			newStatementCount = 0;
		}

		// Set the completion statistics.
		updateStatistics(chunkContext.getStepContext().getStepExecution().getJobExecution(), newStatementCount);

		return successful;
	}

}
