package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.MEMORY_OPEN_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.READ_FAILURE_STATUS;

import java.util.List;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemReadListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;
import org.springframework.batch.item.support.AbstractItemCountingItemStreamItemReader;
import org.springframework.util.ClassUtils;

import gov.va.cpss.model.fps.PSPatient;

/**
 * Implementation of AbstractItemCountingItemStreamItemReader used by the Generate CBS
 * batch job to read data from an in-memory map of CBS Account ID to list of PSPatient records.
 * 
 * @author DNS   
 */
public class CbsPatientMapItemReader<T> extends AbstractItemCountingItemStreamItemReader<Entry<Long, List<PSPatient>>>
		implements ItemReadListener<Entry<Long, List<PSPatient>>>, StepExecutionListener {

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt read.
	 */
	private boolean forceStop = false;

	/*
	 * Flag to indicate the map has been opened.
	 */
	private boolean opened = false;

	private CbsRuntimeState cbsRuntimeState;

	private JobExecution jobExecution;

	private final Logger readerLogger;

	/*
	 * The count of entries read.
	 */
	private int lineCount = 0;

	public CbsPatientMapItemReader() {
		setName(ClassUtils.getShortName(CbsPatientMapItemReader.class));
		readerLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public CbsRuntimeState getCbsRuntimeState() {
		return cbsRuntimeState;
	}

	public void setCbsRuntimeState(CbsRuntimeState cbsRuntimeState) {
		this.cbsRuntimeState = cbsRuntimeState;
	}

	@Override
	protected void doOpen() throws Exception {

		opened = false;

		if (cbsRuntimeState.isEmpty()) {

			stopJob(MEMORY_OPEN_ERROR_STATUS,
					"Unable to open memory map for reading because there was no data detected");

		} else if (cbsRuntimeState.isDataError()) {

			stopJob(MEMORY_OPEN_ERROR_STATUS,
					"Unable to open memory map for reading because data error was encountered");
		} else {

			opened = true;
		}
	}

	@Override
	public Entry<Long, List<PSPatient>> read() throws Exception, UnexpectedInputException, ParseException {

		// Do not attempt to read if forced stop.
		if (forceStop) {
			readerLogger.error("Unrecoverable system error forcing reader to stop processing");
			return null;
		}

		try {

			return super.read();

		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Unable to read item because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			stopJob(READ_FAILURE_STATUS, error.toString());
		}

		// Null response will cause step to end.
		return null;
	}

	@Override
	protected Entry<Long, List<PSPatient>> doRead() throws Exception {

		// If not opened then return null which will effectively stop
		// processing.
		if (!opened) {
			return null;
		}

		// If null is read then return null which will effectively stop
		// processing.
		return readEntry();
	}

	/**
	 * Read the entry from the map.
	 * 
	 * @return The next entry in the map.
	 */
	private Entry<Long, List<PSPatient>> readEntry() {

		Entry<Long, List<PSPatient>> entry = cbsRuntimeState.pollCurrentPSPatientM();

		if (entry == null) {
			return null;
		}

		lineCount++;

		return entry;
	}

	@Override
	protected void doClose() throws Exception {

		// Reset the stop flag on close.
		forceStop = false;

		// Reset the open flag on close.
		opened = false;

		// Be sure to reset the line count.
		lineCount = 0;
	}

	@Override
	public void beforeRead() {
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			readerLogger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void onReadError(Exception e) {

		readerLogger.error("Reader encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to read item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		error.append("\nCause: ");
		error.append(e.getCause().getMessage().trim());

		// Set failure and message.
		stopJob(READ_FAILURE_STATUS, error.toString());
	}

	@Override
	public void afterRead(Entry<Long, List<PSPatient>> item) {
		// Do not need to do anything after read.
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		readerLogger.info("Before Step Execution");
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set exit status if a failure
		// during read processing.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		readerLogger.info("After Step Execution");
		// Do not do anything special here.
		return null;
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		readerLogger.error("Reader execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		readerLogger.error("Read failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		readerLogger.error("Read failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
