package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.DATA_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_ERROR_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_ERROR_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;

import java.util.List;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ItemProcessListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.annotation.AfterStep;
import org.springframework.batch.core.annotation.BeforeStep;
import org.springframework.batch.item.ItemProcessor;

import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.service.CbsService;

/**
 * Custom ItemProcessor used by the Generate CBS Data batch job to consolidate statements
 * in a memory map of CBS Account ID to list of associated PSPatient records.
 * 
 * @author DNS   
 */
public class CbsPatientEntryItemProcessor implements ItemProcessor<Entry<Long, List<PSPatient>>, CBSStmt>,
		ItemProcessListener<Entry<Long, List<PSPatient>>, CBSStmt> {

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt processing.
	 */
	private boolean forceStop = false;

	private final Logger processorLogger;

	private JobExecution jobExecution;

	private Long batchRunId;

	private CbsService cbsService;

	public CbsPatientEntryItemProcessor() {
		processorLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	@Override
	public CBSStmt process(Entry<Long, List<PSPatient>> pojo) throws Exception {

		processorLogger.debug("Begin Process");

		// Do not attempt to process if forced stop.
		if (forceStop) {
			return null;
		}

		final long cbssAccount = pojo.getKey();

		processorLogger.debug("Processing patient record list for CBS Account: " + cbssAccount);

		CBSStmt statement = null;

		// At this point batchRunId FK should not be null.
		if (batchRunId != null) {

			try {

				statement = cbsService.buildStatementForPatientMap(batchRunId, pojo);

				// If the statement is null then just continue processing.
				if (statement == null) {

					setDataErrorAndMessage(
							"No statement can be generated for CBS Account (" + cbssAccount + ") but will continue processing");
				}

			} catch (Exception e) {

				// Unrecoverable error so stop the job.
				stopJob(PROCESSING_FAILURE_STATUS, "Unrecoverable processing error: " + e.getMessage());
			}

		} else {

			// Unrecoverable error so stop the job.
			stopJob(PROCESSING_FAILURE_STATUS, "Attempted to process with a null Batch Run FK");
		}

		processorLogger.debug("End Process");

		// If flagged to forcefully stop then return null which will effectively
		// stop processing.
		if (forceStop) {
			return null;
		}

		// Return the statement.
		return statement;
	}

	/**
	 * Before processing capture the job execution and the received id for data
	 * processing.
	 * 
	 * @param stepExecution
	 */
	@BeforeStep
	public void beforeStep(StepExecution stepExecution) {
		processorLogger.info("Before Step Execution");

		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();

		// Save the batchRunId at the beginning of the step.
		// It is obtained by the batch prior to the job and passed as a job
		// parameter when the job starts.
		batchRunId = jobExecution.getJobParameters().getLong(BATCH_RUN_ID_KEY);
	}

	/**
	 * After processing check for unexpected conditions that suggest an error.
	 * 
	 * @param stepExecution
	 */
	@AfterStep
	public void afterStep(StepExecution stepExecution) {

		processorLogger.info("After Step Execution");
	}

	@Override
	public void beforeProcess(Entry<Long, List<PSPatient>> item) {
		processorLogger.debug("Before Process");
		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			processorLogger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterProcess(Entry<Long, List<PSPatient>> item, CBSStmt result) {
		// Do not need to do anything after process.
	}

	@Override
	public void onProcessError(Entry<Long, List<PSPatient>> item, Exception e) {
		processorLogger.error("Processor encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to process item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		error.append("\nCause: ");
		error.append(e.getCause().getMessage().trim());

		// Set failure and message.
		stopJob(PROCESSING_FAILURE_STATUS, "Unrecoverable processor error");
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @return Return a null record to stop step processing.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		processorLogger.error("Processor execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the data error and message in the job execution context. This is an
	 * error condition that allows continuing processing.
	 * 
	 * @param message
	 *            The message to associate with the data error status.
	 */
	private void setDataErrorAndMessage(final String message) {

		// Log job error status.
		processorLogger.error("Job error with status: " + DATA_ERROR_STATUS);

		// Log job error message.
		processorLogger.error("Job error message: " + message);

		// Set job error.
		jobExecution.getExecutionContext().putString(JOB_ERROR_KEY, DATA_ERROR_STATUS);

		// Append message to job context if already exists.
		String existingErrorMessage = null;
		StringBuffer errorMessageBuff = new StringBuffer();

		// Add (append if necessary) to the job execution.
		if (jobExecution.getExecutionContext().containsKey(JOB_ERROR_MESSAGE_KEY)) {
			existingErrorMessage = jobExecution.getExecutionContext().getString(JOB_ERROR_MESSAGE_KEY);
		}
		if (existingErrorMessage != null) {
			errorMessageBuff.append(existingErrorMessage);
		}
		// Message should always be unique so always append.
		errorMessageBuff.append(message);

		// Set job error message.
		jobExecution.getExecutionContext().putString(JOB_ERROR_MESSAGE_KEY, message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		processorLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		processorLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
