package gov.va.cpss.job.cbs;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY;

import static java.util.stream.Collectors.toList;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;

import gov.va.cpss.model.cbs.CBSSitePatient;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.service.CbsService;

/**
 * Implementation of ItemWriter used to handle writing of consolidated
 * statements when processing.
 * 
 * @author DNS  
 */
public class CbsDelegateMultiItemWriter
		implements ItemWriter<CBSStmt>, ItemWriteListener<CBSStmt>, StepExecutionListener {

	private final Logger writerLogger = Logger.getLogger(CbsDelegateMultiItemWriter.class.getCanonicalName());

	/*
	 * Flag to indicate that the job has been forcefully stopped and should no
	 * longer attempt writes.
	 */
	private boolean forceStop = false;

	/*
	 * Count of statements written successfully.
	 */
	private long initialStatementCount = 0;

	/*
	 * Count of replaced statements updated successfully.
	 */
	private long replacedStatementCount = 0;

	private JobExecution jobExecution;

	private CbsService cbsService;

	/*
	 * Delegate writer used to take advantage of batch updating replaced
	 * statements.
	 */
	private ItemWriter<Long> cbsStmtDatabaseItemUpdater;

	/*
	 * Delegate writer used to take advantage of batch writing site transaction
	 * records.
	 */
	private ItemWriter<CBSSiteTrans> cbsSiteTransDatabaseItemWriter;

	public CbsService getCbsService() {
		return cbsService;
	}

	public void setCbsService(CbsService cbsService) {
		this.cbsService = cbsService;
	}

	public ItemWriter<Long> getCbsStmtDatabaseItemUpdater() {
		return cbsStmtDatabaseItemUpdater;
	}

	public void setCbsStmtDatabaseItemUpdater(ItemWriter<Long> cbsStmtDatabaseItemUpdater) {
		this.cbsStmtDatabaseItemUpdater = cbsStmtDatabaseItemUpdater;
	}

	public ItemWriter<CBSSiteTrans> getCbsSiteTransDatabaseItemWriter() {
		return cbsSiteTransDatabaseItemWriter;
	}

	public void setCbsSiteTransDatabaseItemWriter(ItemWriter<CBSSiteTrans> cbsSiteTransDatabaseItemWriter) {
		this.cbsSiteTransDatabaseItemWriter = cbsSiteTransDatabaseItemWriter;
	}

	@SuppressWarnings({ "unused", "unchecked" })
	@Override
	public void write(List<? extends CBSStmt> items) throws Exception {

		writerLogger.debug("Begin Write");

		// Do not attempt to write if forced stop.
		if (forceStop) {
			return;
		}

		try {
			// Track list of replaced statements.
			List<Long> replacedStatementL = new ArrayList<>();

			if (true) {
				// NEW WAY - using batch inserts but we don't have
				// access to the new primary key id fields
				cbsService.batchSaveStatements((List<CBSStmt>) items);
				
				initialStatementCount += items.size();

				replacedStatementL = items.stream().map(cbsStmt -> cbsStmt.getReplacedStmtId())
						.filter(replacedStmtId -> replacedStmtId != 0).collect(toList());
			} else {
				// OLD WAY

				// Loop through all of the statements to write.
				for (Object item : items) {

					CBSStmt statement = (CBSStmt) item;

					// Batch won't return generated keys so use DAO.
					writerLogger.debug("Saving CBSStmt: " + statement.toString());
					cbsService.saveStatement(statement);

					// Increment the write count for tabulating expected totals
					// during final completion task.
					++initialStatementCount;

					// Loop through all of the SiteStmt.
					for (CBSSiteStmt siteStatement : statement.getSiteStmtL()) {

						// Save the database auto-generated statement ID in the
						// site
						// statement.
						siteStatement.setStmtId(statement.getId());

						// Batch won't return generated keys so use service.
						writerLogger.debug("Saving CBSSiteStmt");
						cbsService.saveSiteStatement(siteStatement);

						// Update the site patient with the associated site
						// statement ID.
						writerLogger.debug("Saving CBSSitePatient");
						CBSSitePatient sitePatient = siteStatement.getSitePatient();
						sitePatient.setSiteStmtId(siteStatement.getId());

						// Since one to one ratio to site statement just use the
						// service to save.
						cbsService.savePatient(sitePatient);

						// Update the list of site trans with the associated
						// site
						// statement ID.
						List<CBSSiteTrans> siteTransL = siteStatement.getSiteTransL();
						siteTransL.stream().forEach(u -> u.setSiteStmtId(siteStatement.getId()));

						// We don't need generated keys so we can use the batch
						// for
						// this insert.
						writerLogger.debug("Saving list of CBSSiteTrans");
						cbsSiteTransDatabaseItemWriter.write(siteTransL);

					}

					// Append to replaced statement list if necessary.
					if (statement.getReplacedStmtId() != 0) {
						replacedStatementL.add(statement.getReplacedStmtId());
					}

				}

			}

			// We can use batch to update replaced statements.
			if (!replacedStatementL.isEmpty()) {
				writerLogger.debug("Updating Replaced Statements");
				cbsStmtDatabaseItemUpdater.write(replacedStatementL);
				// Save the count for tabulating expected totals during final
				// completion task.
				replacedStatementCount = replacedStatementL.size();
			}

		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Error during write because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			// Unrecoverable error so stop the job.
			stopJob(WRITE_FAILURE_STATUS, error.toString());
		}

		writerLogger.debug("End Write");
	}

	@Override
	public void beforeWrite(List<? extends CBSStmt> items) {
		// Reset the write count.
		initialStatementCount = 0;
		replacedStatementCount = 0;

		forceStop = false;
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
			writerLogger.error("System failure detected.");
			forceStop = true;
		}
	}

	@Override
	public void afterWrite(List<? extends CBSStmt> items) {

		// Saving running total of the written counts in the job execution
		// context for completion task reference.
		if ((items != null) && !items.isEmpty()) {

			if (!updateInitialStatementCount()) {
				// Set failure and message.
				stopJob(WRITE_FAILURE_STATUS, "Unable to update running total initial statement count");
			} else if (!updateReplacedStatementCount()) {
				// Set failure and message.
				stopJob(WRITE_FAILURE_STATUS, "Unable to update running total replaced statement count");
			}
		}
	}

	/**
	 * Update the running total of initial statements written in the job
	 * execution context.
	 * 
	 * @return Boolean flag indicating if successful or not.
	 */
	private boolean updateInitialStatementCount() {

		boolean successful = false;

		if (jobExecution.getExecutionContext().containsKey(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY)) {
			jobExecution.getExecutionContext().putLong(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY,
					jobExecution.getExecutionContext().getLong(CBS_TOTAL_INITIAL_STATEMENT_COUNT_KEY)
							+ initialStatementCount);
			successful = true;
		}

		return successful;
	}

	/**
	 * Update the running total of replaced statements updated in the job
	 * execution context.
	 * 
	 * @return Boolean flag indicating if successful or not.
	 */
	private boolean updateReplacedStatementCount() {

		boolean successful = false;

		if (jobExecution.getExecutionContext().containsKey(CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY)) {
			jobExecution.getExecutionContext().putLong(CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY,
					jobExecution.getExecutionContext().getLong(CBS_TOTAL_REPLACED_STATEMENT_COUNT_KEY)
							+ replacedStatementCount);
			successful = true;
		}

		return successful;
	}

	@Override
	public void onWriteError(Exception e, List<? extends CBSStmt> items) {

		writerLogger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();
		error.append("Unable to write item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());
		if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, "Unrecoverable writer error");
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		writerLogger.info("Before Step Execution");
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set key values as data is
		// processed.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		writerLogger.info("After Step Execution");
		// Do not need to do anything after step.
		return null;
	}

	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	private void stopJob(final String status, final String message) {

		// Set the flag to indicate the job has been forcefully stopped.
		forceStop = true;

		// Log message.
		writerLogger.error("Writer execution encountered unrecoverable error and forced stop");

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	private void setFailureStatus(final String status) {

		// Log job failure status.
		writerLogger.error("Job failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	private void setFailureMessage(final String message) {

		// Log job failure message.
		writerLogger.error("Job failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

}
