package gov.va.cpss.job.caps;

import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;
import static gov.va.cpss.model.ps.Constants.BATCH_PROCESS_ID_KEY;
import static gov.va.cpss.model.ps.Constants.RECEIVED_IDS_KEY;

import java.util.List;
import java.util.Map.Entry;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;

import gov.va.cpss.job.CBSSPagingMultiItemWriter;

import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.apps.APSSiteStmt;

import gov.va.cpss.model.ProcessStatus.Status;

import gov.va.cpss.service.GenerateAPPSService;

/**
 * 
 * Implementation of ItemWriter used to handle writing of raw records when
 * generating APPS data.
 * 
 * Copyright HPE / VA
 * January 25, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class CAPSPagingMultiItemWriter extends CBSSPagingMultiItemWriter<Entry<APSStmt, List<APSSiteStmt>>> 
{
	// Injected property
	private GenerateAPPSService generateAPPSService;

	public GenerateAPPSService getGenerateAPPSService()
	{
		return this.generateAPPSService;
	}

	public void setGenerateAPPSService(GenerateAPPSService inGenerateAPPSService)
	{
		this.generateAPPSService = inGenerateAPPSService;
	}

    @Override
    protected boolean doWrite(List<? extends Entry<APSStmt, List<APSSiteStmt>>> inItems)
    {
        this.logger.debug("Do writing.");

        String message = this.generateAPPSService.save(inItems);

        if (message != null)
        {
            stopJob(WRITE_FAILURE_STATUS, message);
            return false;
        }

        return true; 
    }

	@Override
	public ExitStatus afterStep(StepExecution stepExecution)
	{
        this.logger.debug("After Step - Last write / update.");

        String message = null;

        ExitStatus status = super.afterStep(stepExecution);

        long processId = stepExecution.getJobExecution().getJobParameters().getLong(BATCH_PROCESS_ID_KEY).longValue();

	    if (status.equals(ExitStatus.FAILED))
	    {
	        // Remove unfinished statements and all its related data.
	        message = this.generateAPPSService.deleteStatements(processId);

	        if (message != null)
	        {
	            this.logger.error(message);
	        }

	        return status;
	    }

	    @SuppressWarnings("unchecked")
        List<Long> apsReceivedIds = (List<Long>) stepExecution.getExecutionContext().get(RECEIVED_IDS_KEY);

	    // Update Statement status to NEW, and APSReceived status to PROCESSED.
	    message = this.generateAPPSService.updateStatementsStatus(processId, Status.NEW);

	    if (message != null)
	    {
	        this.logger.error(message);

	        return ExitStatus.FAILED;
	    }

	    message = this.generateAPPSService.updateAPSReceivedStatus(apsReceivedIds, Status.PROCESSED);

	    if (message != null)
	    {
	        this.logger.error(message);

	        return ExitStatus.FAILED;
	    }

	    // Set Received IDs to null for GC.
	    stepExecution.getExecutionContext().put(RECEIVED_IDS_KEY, null);

	    return ExitStatus.COMPLETED;
	}
}
