package gov.va.cpss.job.caps;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_ERROR_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;

import static gov.va.cpss.model.ps.Constants.TOTAL_READ_COUNT_KEY;
import static gov.va.cpss.model.ps.Constants.AVAILABLE_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.BATCH_PROCESS_ID_KEY;
import static gov.va.cpss.model.ps.Constants.GENERATED_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.TOTAL_CS_INITIAL_COUNT_KEY;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobExecutionListener;

import gov.va.cpss.model.ProcessStatus.Status;

import gov.va.cpss.service.GenerateAPPSService;

/**
 * Custom JobExecutionListener used to process custom exit status to handle
 * errors during Generate APPS batch job execution.
 * 
 * Copyright HPE / VA
 * January 17, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class CAPSJobExecutionListener implements JobExecutionListener
{
    private Log logger = LogFactory.getLog(getClass());

	private GenerateAPPSService generateAPPSService;

	public GenerateAPPSService getGenerateAPPSService()
	{
		return this.generateAPPSService;
	}

	public void setGenerateAPPSService(GenerateAPPSService inGenerateAPPSService)
	{
		this.generateAPPSService = inGenerateAPPSService;
	}

    @Override
	public void beforeJob(JobExecution jobExecution)
	{
		this.logger.debug("Beginning Job Execution");

		// Initialize written running totals.
		jobExecution.getExecutionContext().putLong(TOTAL_CS_INITIAL_COUNT_KEY, 0);
	}

	@Override
	public void afterJob(JobExecution jobExecution)
	{
		this.logger.debug("After Job");

		// System failure takes precedence over data error.
		if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
		{
			final String exitStatus = jobExecution.getExecutionContext().getString(JOB_FAILURE_KEY);

			if (jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY))
			{
				final String exitMessage = jobExecution.getExecutionContext().getString(JOB_FAILURE_MESSAGE_KEY);

				this.logger.error("JobFailure detected, setting exit status (" + exitStatus + "): " + exitMessage);

				jobExecution.setExitStatus(new ExitStatus(exitStatus, exitMessage));
			}
			else
			{
				this.logger.error("JobFailure detected, setting exit status (" + exitStatus + ")");

				jobExecution.setExitStatus(new ExitStatus(exitStatus));
			}

		}
		else if (jobExecution.getExecutionContext().containsKey(JOB_ERROR_KEY))
		{
			final String exitStatus = jobExecution.getExecutionContext().getString(JOB_ERROR_KEY);

			this.logger.error("JobError detected, setting exit status (" + exitStatus + ")");

			jobExecution.setExitStatus(new ExitStatus(exitStatus));
		}

		// Set the completion statistics.
		setCompletionStatistics(jobExecution);

		this.logger.debug("Ending Job Execution.");
	}

	/**
	 * Set the completion statistics.
	 * 
	 * @param jobExecution
	 *            The job execution to append statistics to.
	 */
	private void setCompletionStatistics(JobExecution jobExecution)
	{
	    Long processId = jobExecution.getJobParameters().getLong(BATCH_PROCESS_ID_KEY);

	    // Number of total Statements generated.
		final Long generatedStmtsCount = this.generateAPPSService.getStatementCountWithStatus(processId.longValue(), Status.NEW);

		if (generatedStmtsCount != null)
		{
			jobExecution.getExecutionContext().putString(GENERATED_CS_STATISTICS_KEY, generatedStmtsCount.toString());
		}
		else
		{
			jobExecution.getExecutionContext().putString(GENERATED_CS_STATISTICS_KEY, "Generated Consolidated Statements Statistics Unavailable");
		}

		// Number of total Site Patients read / loaded.
		final long totalReadCount = jobExecution.getExecutionContext().getLong(TOTAL_READ_COUNT_KEY);
		jobExecution.getExecutionContext().putString(AVAILABLE_CS_STATISTICS_KEY, String.valueOf(totalReadCount));
	}
}
