package gov.va.cpss.job.appsprintack;

import java.beans.PropertyEditorSupport;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.item.file.mapping.DefaultLineMapper;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;

import gov.va.cpss.model.appsprintack.APPSADFileRecord;
import gov.va.cpss.model.appsprintack.AppsDetailListField;

/**
 * Editor for parsing the Detail portion (AD) of PA records. Expects a list of
 * AD records delimited by the specified recordDelimiter, default |. Returns a
 * List of APPSADFileRecord objects parsed from the AD records.
 * 
 * @author Andrew Vance
 */
public class AppsDetailListFieldEditor extends PropertyEditorSupport implements InitializingBean {

	public static String DEFAULT_RECORD_DELIMITER = "}";

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());;

	private String recordDelimiter = DEFAULT_RECORD_DELIMITER;

	private DefaultLineMapper<APPSADFileRecord> appsAdFileRecordLineMapper;

	public String getRecordDelimiter() {
		return recordDelimiter;
	}

	public void setRecordDelimiter(String recordDelimiter) {
		this.recordDelimiter = recordDelimiter;
	}

	public DefaultLineMapper<APPSADFileRecord> getAdFileRecordLineMapper() {
		return appsAdFileRecordLineMapper;
	}

	public void setAdFileRecordLineMapper(DefaultLineMapper<APPSADFileRecord> adFileRecordLineMapper) {
		this.appsAdFileRecordLineMapper = adFileRecordLineMapper;
	}

	@Override
	public void setAsText(String text) {
		logger.debug("Parsing:" + text);
		try {
			final List<APPSADFileRecord> printAckDetailL = new ArrayList<APPSADFileRecord>();

			if (text != null) {
				String[] adRecords = text.split(recordDelimiter);
				logger.debug("Number of fields found:" + adRecords.length);

				int lineNumber = 1;
				for (String adRecord : adRecords) {
					printAckDetailL.add(appsAdFileRecordLineMapper.mapLine(adRecord, lineNumber++));
				}
			}

			setValue(new AppsDetailListField() {
				@Override
				public List<APPSADFileRecord> getPrintAckDetailL() {
					return printAckDetailL;
				}
			});
		} catch (Exception e) {
			logger.error("Caught exception: " + e.getMessage());
			throw new RuntimeException(e);
		}
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		Assert.notNull(appsAdFileRecordLineMapper, "The APPSADFileRecordLineMapper must be set");
	}
}
