package gov.va.cpss.job.apps;

import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;

import static gov.va.cpss.model.ps.Constants.MESSAGE_RESOURCE_KEY;
import static gov.va.cpss.model.ps.Constants.MESSAGE_DIRECTORY_KEY;

import java.io.File;


import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;

import org.springframework.batch.repeat.RepeatStatus;

import gov.va.cpss.job.CBSSBaseFileTransferTasklet;

/**
 * 
 * Complete Load APPS processing by transferring the error file to SFTP server
 * with temporary name, renaming temporary file on SFTP server with correct name.
 * If any errors occur the batch job status is set to error condition.
 * 
 * Copyright DXC / VA
 * April 21, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public class APPSFileTransferTasklet extends CBSSBaseFileTransferTasklet
{
    @Override
    public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception
    {
        boolean successful = true;

        this.logger.debug("File Transfer Tasklet stated.");

        // Save the jobExecution for reference.
        JobExecution jobExecution = chunkContext.getStepContext().getStepExecution().getJobExecution();

        // Obtain the temporary local file paths as they will be needed regardless of failure.
        // Annual statements file local path
        final String stageFilename = getJobParameter(jobExecution, MESSAGE_RESOURCE_KEY);
        final String serverTargetDirectory = getJobParameter(jobExecution, MESSAGE_DIRECTORY_KEY);

        if ((stageFilename == null) || stageFilename.isEmpty())
        {
            return RepeatStatus.FINISHED;
        }

        File stageFile = new File(getStagingDirectory(), stageFilename);

        if (!stageFile.exists())
        {
            return RepeatStatus.FINISHED;
        }

        // Send message file with temporary filename (with timestamp extension).
        // Example:
        // 'APPS-06022016.txt' would be temporarily sent as
        // 'APPS-06022016.txt.<timestamp>.loadapps' 
        final String tempFilename = getTemporaryFilename(stageFilename);

        // Ftp file to server
        if (this.sftpService.ftpFileToServerWithName(stageFile.getAbsolutePath(), tempFilename, serverTargetDirectory))
        {
            this.logger.debug("Ftp file to server sucessfully: from " + stageFile.getAbsolutePath() + " to: /" + serverTargetDirectory + "/" + tempFilename);
        }
        else
        {
            // This is an unexpected and unrecoverable error.
            final String error = "Unsuccessful file transfer: " + getTemporaryFilename(stageFilename);
            setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

            successful = false;
        }

        // Rename file on server
        if (successful)
        {
            String messageFileName = stageFilename.replace(".txt", ".err");

            if (this.sftpService.ftpRenameFileInDirectory(tempFilename, messageFileName, serverTargetDirectory))
            {
                this.logger.debug("Renamed file on server sucessfully: " + messageFileName);
            }
            else
            {
                // This is an unexpected and unrecoverable error.
                final String error = "Unsuccessful file rename: " + stageFilename;
                setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);

                successful = false;
            }
        }

        // Regardless of failure or successful batch run, delete local statement files if exist.
        if (deleteResource(stageFile.getAbsolutePath()))
        {
            this.logger.debug("Deleted temporary local file sucessfully: " + stageFile.getAbsolutePath());
        }
        else
        {
            // Don't make this a failure in case transfer had already been
            // successful.
            this.logger.warn("Unable to delete the temporary local file: " + stageFile.getAbsolutePath());
        }

        if(!successful)
        {
            final String error = "Unsuccessful file transfer: " + stageFilename;
            setFailureStatus(jobExecution, WRITE_FAILURE_STATUS, error);
        }

        this.logger.debug("File Transfer Tasklet ended.");

        return RepeatStatus.FINISHED;
    }

}
