package gov.va.cpss.job.apps;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
//import static gov.va.cpss.job.CbssJobProcessingConstants.EMPTY_FILE_ERROR_STATUS;

import static gov.va.cpss.model.ps.Constants.RECEIVED_ID_KEY;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;

import gov.va.cpss.job.CBSSDelegateProcessor;

import gov.va.cpss.model.apps.APPSCompositeData;
import gov.va.cpss.model.apps.APSSite;

/**
 *
 * A custom composite / delegate processor for Load APPS batch job,
 * which will send the data to different processors, depending on
 * whether the data have errors or not.
 * 
 * Copyright DXC / VA
 * April 12, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
//@SuppressWarnings("nls")
public class APPSDelegateProcessor extends CBSSDelegateProcessor<APPSCompositeData, Entry<APSSite, ? extends List<?>>>
{
    private Long receivedId;

    /**
     * Before processing capture the job execution and the received id for data
     * processing.
     * 
     * @param stepExecution
     */
    @Override
    public void beforeStep(StepExecution stepExecution)
    {
        super.beforeStep(stepExecution);

        // Save the receivedId at the beginning of the step.
        // It is obtained by the batch prior to the job and passed as a job
        // parameter when the job starts.
        this.receivedId = this.jobExecution.getJobParameters().getLong(RECEIVED_ID_KEY);
    }

    /**
     * After processing check for unexpected conditions that suggest an error.
     * 
     * @param stepExecution
     */
    @Override
    public ExitStatus afterStep(StepExecution stepExecution)
    {
        return super.afterStep(stepExecution);
    }

    @Override
    public Entry<APSSite, ? extends List<?>> process(APPSCompositeData compositeData) throws Exception
    {
        // If job has already failed then process and return error message.
        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            String message = this.jobExecution.getExecutionContext().getString(JOB_FAILURE_MESSAGE_KEY);
            List<String> messages = new ArrayList<>();
            messages.add(message);

            if (compositeData == null)
            {
                APPSCompositeData messageCompositeData = new APPSCompositeData();
                messageCompositeData.setMessageEntry(new AbstractMap.SimpleEntry<>(null, messages));
                messageCompositeData.setValid(false);

                return super.process(messageCompositeData);
            }

            if (compositeData.getMessageEntry() != null && compositeData.getMessageEntry().getValue() != null)
            {
                compositeData.getMessageEntry().getValue().add(message);
            }
            else
            {
                compositeData.setMessageEntry(new AbstractMap.SimpleEntry<>(null, messages));
                compositeData.setValid(false);
            }

            return super.process(compositeData);
        }

        if (this.receivedId != null)
        {
            //this.logger.debug("Received ID: " + this.receivedId);

            if (compositeData.getSiteEntry() != null)
            {
                compositeData.getSiteEntry().getKey().setApsReceivedId(this.receivedId.longValue());
            }

            if (compositeData.getMessageEntry() != null)
            {
                compositeData.getMessageEntry().getKey().setApsReceivedId(this.receivedId.longValue());
            }
        }

        return super.process(compositeData);
    }
}
