package gov.va.cpss.job;

import java.sql.Timestamp;
import java.util.Calendar;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_TIMESTAMP_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;

import static gov.va.cpss.job.updateicn.UpdateICNConstants.ICN_UPDATE_SUCCESS_KEY;
import static gov.va.cpss.job.updateicn.UpdateICNConstants.TOTAL_ICN_UPDATED_KEY;
import static gov.va.cpss.job.updateicn.UpdateICNConstants.ICN_UPDATE_MESSAGE_KEY;

import gov.va.cpss.model.BatchRun;

/**
 * 
 * This is a scheduled batch job that will invoke Update ICN Service
 * to update ICN's throughout the CBSS, based on business rules and requirements.
 * 
 * @author Yiping Yao
 * @version 1.0 - 20160803
 *
 */
@SuppressWarnings("nls")
public class UpdateICNJob extends CbssBaseJob
{
	@Autowired
	private FlowJob UpdateICNJobBatch;

	@Override
	protected FlowJob getJob()
	{
		return this.UpdateICNJobBatch;
	}

	@Override
	protected boolean runJob(final BatchRun batchRun, FlowJob flowJob)
	{
		jobLogger.info("Start UpdatICNJob.runJob.");

		boolean isSuccessful = true;
		String errorMsg;

		Timestamp batchRunTimestamp = new Timestamp(Calendar.getInstance().getTime().getTime());

		try
		{
			JobParameters parameters = getParameters(batchRun.getId(), batchRunTimestamp);

			JobExecution execution = executeJob(flowJob, parameters);

			if (execution == null)
			{
				isSuccessful = false;
				errorMsg = "Processing ended with null execution";
				logExecuteJobException(errorMsg);
				appendErrorMessage(errorMsg);

				jobLogger.debug(errorMsg);
			}
			else if (execution.getExitStatus() != ExitStatus.COMPLETED)
			{
				isSuccessful = false;

				if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
				{
					errorMsg = "Processing ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY);
				}
				else
				{
					errorMsg = "Processing ended with unknown error: " + execution.getExitStatus();
				}

				logExecuteJobException(errorMsg);
				appendErrorMessage(errorMsg);

				jobLogger.debug(errorMsg);
			}

			if (execution != null)
			{
				isSuccessful = ((Boolean) execution.getExecutionContext().get(ICN_UPDATE_SUCCESS_KEY)).booleanValue();

				if (isSuccessful)
				{
					buildInfoMessage( String.valueOf(execution.getExecutionContext().getLong(TOTAL_ICN_UPDATED_KEY)),
									  execution.getExecutionContext().getString(ICN_UPDATE_MESSAGE_KEY) );
				}
				else
				{
					errorMsg = execution.getExecutionContext().getString(ICN_UPDATE_MESSAGE_KEY);
					appendErrorMessage(errorMsg);

					jobLogger.debug(errorMsg);
				}
			}
		}
		catch (Exception e)
		{
			isSuccessful = false;
			errorMsg = e.getMessage();
			e.printStackTrace();
			logExecuteJobException(errorMsg);
			super.appendErrorMessage(errorMsg);
			
			jobLogger.debug(errorMsg);
		}

		jobLogger.info("End UpdatICNJob.runJob.");

		return isSuccessful;
	}

	/**
	 * Get Job Parameters
	 * 
	 * @param batchRunId
	 * @param batchRunTimestamp
	 * @return
	 */
	@SuppressWarnings("static-method")
	private JobParameters getParameters(final int batchRunId, final Timestamp batchRunTimestamp)
	{
		// Build job parameters for the batchRunId and batchRunTimestamp
		return new JobParametersBuilder().addLong(BATCH_RUN_ID_KEY, new Long(batchRunId))
										 .addLong(BATCH_RUN_TIMESTAMP_KEY, new Long(batchRunTimestamp.getTime())).toJobParameters();
	}

	/**
	 * Build this job specific informational message.
	 */
	private void buildInfoMessage(String totalICNUpdated, String message)
	{
		StringBuffer strBuff = new StringBuffer();

		strBuff.append("\n");
		strBuff.append(message);
		strBuff.append("\n");
		strBuff.append("Total ICN's updated: ").append(totalICNUpdated).append("\n");
	
		appendInfoMessage(strBuff.toString());
	}
}