package gov.va.cpss.job;

import java.util.Random;

import org.apache.log4j.Logger;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.QuartzJobBean;

import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.service.BatchService;
import gov.va.cpss.service.EmailService;

/**
 * Base job stub class for CBSS quartz jobs for temporary demo purposes.
 * Eventually all implementing classes should be converted to extend CbssBaseJob.  
 * @author DNS  
 */
public abstract class SimpleTestJob extends QuartzJobBean {

	protected static final Logger jobLogger = Logger.getLogger(SimpleTestJob.class.getCanonicalName());
	
	/**
	 * The batch job name that is populated from the database at scheduler
	 * initialization.
	 */
	protected String name;
	
	/**
	 * The service used to manage batch issues.
	 */
	@Autowired
	private BatchService batchService;

	/**
	 * The service used to send email.
	 */
	@Autowired
	protected EmailService emailService;
	
	/*
	 * The error message for the batch run.
	 */
	protected String errorMessage;

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	protected void executeInternal(JobExecutionContext ctx) throws JobExecutionException {

		// Reset the error message.
		errorMessage = null;

		// Get the batch job.
		BatchJob bj = batchService.getBatchJobByName(name);
		// Start a batch run.
		BatchRun bR = batchService.startRun(bj.getId());

		if (bR != null) {
			jobLogger.info("Job " + name + " Started: " + bR.getStartDate());
			// Run random amount of time in seconds.
			Random randomTime = new Random();
			int sleep = randomTime.nextInt(10);

			// Randomly error the job.
			if ((sleep % 3) == 0) {
				// Simulate Error.
				simulateError(bR);
				if (batchService.errorRun(bR)) {
					jobLogger.info("Job " + name + " Completed with Error: " + bR.getEndDate());
				} else {
					jobLogger.error("Problem Ending Job " + name + " with Error");
				}
			} else {
				// Simulate Job Run.
				try {
					Thread.sleep(sleep * 1000);
				} catch (Exception e) {
					jobLogger.error("Problem Sleeping " + name + " , EX: " + e.getMessage());
				}

				bR.setMessage(buildEmailCustomInfo().toString());
				
				if (batchService.completeRun(bR)) {
					jobLogger.info("Job " + name + " Ended: " + bR.getEndDate());
				} else {
					jobLogger.error("Problem Ending Job " + name);
				}
			}

			emailService.cpssReadyToSendEmail(bj.getEmail(), buildEmailSubject(bj), buildEmailMessage(bj, bR));

		} else {
			jobLogger.error("Problem Starting Job " + name);
			emailService.cpssReadyToSendEmail(bj.getEmail(), buildEmailSubject(bj), "Problem Starting Job " + name);
		}
	}

	protected void simulateError(BatchRun batchRun) {
		errorMessage = "\n\nError Message: \nSimple Error!";
		batchRun.setMessage(errorMessage);
	}

	/**
	 * Build an email subject.
	 * 
	 * @return
	 */
	protected String buildEmailSubject(BatchJob batchJob) {
		return batchJob.getDescription();
	}

	/**
	 * Build an email message that communicates information based on the
	 * specific type of job.
	 */
	protected String buildEmailMessage(BatchJob batchJob, BatchRun batchRun) {
		StringBuffer strBuff = buildEmailCommonInfo(batchJob, batchRun);
		strBuff.append(buildEmailCustomInfo());
		strBuff.append(buildEmailErrorInfo());
		return strBuff.toString();
	}

	/**
	 * Use the BatchRun to return information that every job has in common.
	 */
	protected StringBuffer buildEmailCommonInfo(BatchJob batchJob, BatchRun batchRun) {
		StringBuffer strBuff = new StringBuffer();
		strBuff.append("\nBatch Process Name: ");
		strBuff.append(batchJob.getDescription()); // Read the Job class running
		strBuff.append("\nStart: ");
		strBuff.append(batchRun.getServerStartDateString()); // Read the start
																// date
		strBuff.append("\nEnd: ");
		strBuff.append(batchRun.getServerEndDateString()); // Read the end date
		strBuff.append("\nStatus: ");
		strBuff.append(batchRun.getBatchStatus().getDescription()); // Read the
																	// status
																	// description
		return strBuff;
	}

	/**
	 * Build the error portion of an email.
	 */
	protected StringBuffer buildEmailErrorInfo() {
		StringBuffer strBuff = new StringBuffer();
		if ((errorMessage != null) && !errorMessage.isEmpty()) {
			strBuff.append(errorMessage);
		}
		return strBuff;
	}

	/**
	 * Build the second portion of an email that contains information specific
	 * to the derived job class.
	 */
	abstract StringBuffer buildEmailCustomInfo();
}
