package gov.va.cpss.job;

import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_MESSAGE_ID_KEY;

import java.io.PrintWriter;
import java.io.StringWriter;

import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.SBS_OUTPUT_FILE_NAME_KEY;
import static gov.va.cpss.job.sendcbs.SendCBSProcessingConstants.CBS_TOTAL_PATIENTS_INCLUDED_KEY;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.cbs.CBSMessage;
import gov.va.cpss.service.SendCBSService;

public class SendCBSJob extends CbssBaseJob {

	@Autowired
	private FlowJob SendCBSJobBatch;
	
	@Autowired
	private SendCBSService sendCBSService;

	@Override
	protected FlowJob getJob() {
		return SendCBSJobBatch;
	}

	@Override
	protected boolean runJob(final BatchRun batchRun, FlowJob job) {
		boolean successful = false;

		try {
			CBSMessage cbsMessage = sendCBSService.startIsolatedSendCBSJob(batchRun);
			
			JobParameters parameters = getParameters(cbsMessage);
			JobExecution execution = executeJob(job, parameters);

			successful = sendCBSService.endSendCBSJob(execution, cbsMessage);
			jobLogger.debug("sendCBSService.endSendCBSJob():" + successful);
			
			if (successful) {
				String totalPatientsIndluded = "UNKNOWN";
				
				if (execution.getExecutionContext().containsKey(CBS_TOTAL_PATIENTS_INCLUDED_KEY)) {
					totalPatientsIndluded = ""+execution.getExecutionContext().getLong(CBS_TOTAL_PATIENTS_INCLUDED_KEY);
				} else {
					jobLogger.error("Unable to obtain key from process results: " + CBS_TOTAL_PATIENTS_INCLUDED_KEY);
				}
				buildInfoMessage(totalPatientsIndluded);
			}
		} catch (Exception e) {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			e.printStackTrace(pw);
			jobLogger.error(sw);
			super.appendErrorMessage(e.getMessage());
			successful = false;
		}

		return successful;
	}

	/**
	 * Build parameters for the batch job based on the PSReceived object.
	 * 
	 * @param pR
	 *            The PSReceived object representing the database entry.
	 * @return JobParameters object for this batch job run.
	 */
	protected JobParameters getParameters(CBSMessage cbsMessage) {

		// Build job parameters for the input filename, received primary key id,
		// and the created or modified by string.
		return new JobParametersBuilder().addLong(CBS_MESSAGE_ID_KEY, new Long(cbsMessage.getId()))
				.addString(CBS_OUTPUT_FILE_NAME_KEY, cbsMessage.getCBSFileName())
				.addString(SBS_OUTPUT_FILE_NAME_KEY, cbsMessage.getSBSFileName())
				.toJobParameters();
	}

	/**
	 * Build this job specific informational message.
	 */
	private void buildInfoMessage(String totalPatientsIndluded) {
		StringBuffer strBuff = new StringBuffer();
		
		strBuff.append("\n");
		strBuff.append("Total Patients Included: ").append(totalPatientsIndluded).append("\n");
		strBuff.append("Files: 1\n");

		appendInfoMessage(strBuff.toString());
	}
}