package gov.va.cpss.job;

import static gov.va.cpss.job.fps.FpsProcessingConstants.INPUT_DIRECTORY_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.job.fps.FpsProcessingConstants.RECEIVED_ID_KEY;

import java.io.File;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.fps.PSReceived;
import gov.va.cpss.service.FpsService;

/**
 * CBSS job to Process FPS Data.
 * 
 * @author DNS   
 */
public class ProcessFPSDataJob extends AbstractProcessFileListJob {

	@Autowired
	private FlowJob ProcessFPSDataJobBatch;
	
	/**
	 * The service used to manage job issues.
	 */
	@Autowired
	protected FpsService fpsService;
	
	@Autowired
	private String processFPSServerTargetDirectory;
	
	@Autowired
	private String processFPSServerArchiveTargetDirectory;
	
	@Autowired
	private String processFPSServerErrorTargetDirectory;
	
	@Override
	protected FlowJob getJob() {
		return ProcessFPSDataJobBatch;
	}

	@Override
	protected boolean processFile(String file, int batchRunId, FlowJob job) {
		PSReceived pR = initializeJob(batchRunId);

		JobParameters parameters = getParameters(pR);

		JobExecution execution = executeJob(job, parameters);

		return fpsService.endProcessFpsDataJob(execution, pR);
	}

	/**
	 * Initialize the job by calling the job service to obtain a database entry
	 * representing the PSReceived for this run.
	 * 
	 * @return PSReceived object initialized by the JobService.
	 */
	protected PSReceived initializeJob(final int batchRunId) {

		// Use the job service to start the job.
		return fpsService.startProcessFpsDataJob(batchRunId, (new File(filename)).getName());
	}

	/**
	 * Build parameters for the batch job based on the PSReceived object.
	 * 
	 * @param pR
	 *            The PSReceived object representing the database entry.
	 * @return JobParameters object for this batch job run.
	 */
	protected JobParameters getParameters(PSReceived pR) {

		// Build job parameters for the input filename, received primary key id,
		// and the created or modified by string.
		return new JobParametersBuilder().addString(INPUT_RESOURCE_KEY, filename).addString(INPUT_DIRECTORY_KEY, getDataDirectory())
				.addLong(RECEIVED_ID_KEY, new Long(pR.getId())).toJobParameters();
	}

	@Override
	public String getDataDirectory() {
		return processFPSServerTargetDirectory;
	}

	@Override
	public String getArchiveSubDirectory() {
		return processFPSServerArchiveTargetDirectory;
	}

	@Override
	public String getErrorSubDirectory() {
		return processFPSServerErrorTargetDirectory;
	}

	@Override
	public String getProcessListErrorMessage() {
		return "Unable to connect to the SFTP server to retrieve the patient statement files.\n";
	}

}
