package gov.va.cpss.job;

import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemProcessListener;
import org.springframework.batch.core.ItemReadListener;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;

import gov.va.cpss.performance.SimpleStopWatch;
import gov.va.cpss.performance.StopWatch;

/**
 * StepExecutionListener that logs the start time, end time and elapsed time in
 * milliseconds of a step.
 * 
 * @author Brad Pickle
 *
 */
public class LogExecutionReadProcessWriteListener<T, S>
		implements StepExecutionListener, ItemReadListener<T>, ItemProcessListener<T, S>, ItemWriteListener<S> {

	protected static final Logger logger = Logger
			.getLogger(LogExecutionReadProcessWriteListener.class.getCanonicalName());

	private StopWatch stepStopWatch = new SimpleStopWatch();
	private ItemCounter readCounter = new ItemCounter();
	private ItemCounter betweenReadProcessCounter = new ItemCounter();
	private ItemCounter processCounter = new ItemCounter();
	private ItemCounter betweenProcessWriteCounter = new ItemCounter();
	private ItemCounter writeCounter = new ItemCounter();
	private ItemCounter betweenWriteReadCounter = new ItemCounter();

	// ItemReadListener Interface

	@Override
	public void beforeRead() {
		betweenReadProcessCounter.stopWatch.stop();
		betweenProcessWriteCounter.stopWatch.stop();
		betweenWriteReadCounter.stopWatch.stop();

		readCounter.stopWatch.start();
	}

	@Override
	public void afterRead(T item) {
		readCounter.stopWatch.stop();
		readCounter.itemsProcessed++;

		betweenReadProcessCounter.itemsProcessed++;
		betweenReadProcessCounter.stopWatch.start();
	}

	@Override
	public void onReadError(Exception ex) {
		// Noop
	}

	// ItemProcessListener Interface

	@Override
	public void beforeProcess(T item) {
		betweenReadProcessCounter.stopWatch.stop();
		betweenProcessWriteCounter.stopWatch.stop();

		processCounter.stopWatch.start();
	}

	@Override
	public void afterProcess(T item, S result) {
		processCounter.stopWatch.stop();
		processCounter.itemsProcessed++;

		betweenProcessWriteCounter.itemsProcessed++;
		betweenProcessWriteCounter.stopWatch.start();
	}

	@Override
	public void onProcessError(T item, Exception e) {
		// Noop
	}

	// ItemWriteListener Interface

	@Override
	public void beforeWrite(List<? extends S> items) {
		betweenProcessWriteCounter.stopWatch.stop();

		writeCounter.stopWatch.start();
	}

	@Override
	public void afterWrite(List<? extends S> items) {
		writeCounter.stopWatch.stop();

		if (items != null) {
			writeCounter.itemsProcessed += items.size();
			betweenWriteReadCounter.itemsProcessed += items.size();
		}

		betweenWriteReadCounter.stopWatch.start();
	}

	@Override
	public void onWriteError(Exception exception, List<? extends S> items) {
		// Noop
	}

	// StepExecutionListener Interface

	@Override
	public void beforeStep(StepExecution stepExecution) {
		stepStopWatch.start();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		betweenWriteReadCounter.stopWatch.stop();
		stepStopWatch.stop();
		logStatistics(stepExecution);
		return null;
	}

	/**
	 * Log the elapsed time of the step
	 */
	private void logStatistics(StepExecution stepExecution) {
		logger.info(getMessage(stepExecution, "Reading", readCounter));
		logger.info(getMessage(stepExecution, "Between Reading/Processing", betweenReadProcessCounter));
		logger.info(getMessage(stepExecution, "Processing", processCounter));
		logger.info(getMessage(stepExecution, "Between Processing/Writing", betweenProcessWriteCounter));
		logger.info(getMessage(stepExecution, "Writing", writeCounter));
		logger.info(getMessage(stepExecution, "Between Writing/Reading", betweenWriteReadCounter));
		logger.info(getRemainingTimeMessage(stepExecution));
	}

	/**
	 * Build the log message for the given step and execution point (Start/End)
	 * 
	 * @param stepExecution
	 * @param executionPoint
	 * @return String log message
	 */
	private String getMessage(StepExecution stepExecution, String executionPoint, ItemCounter counter) {
		final StringBuilder s = new StringBuilder();

		s.append("Job:").append(stepExecution.getJobExecution().getJobInstance().getJobName());
		s.append(", Step:").append(stepExecution.getStepName());
		s.append(", Phase:").append(executionPoint);
		s.append(" - Items Processed:").append(counter.itemsProcessed);
		s.append(", Avg Time Per Item:").append(counter.getAverageTimePerItem()).append(" milliseconds");
		s.append(", Total Time:").append(counter.stopWatch.getElapsedTimeAsString());

		return s.toString();
	}

	/**
	 * Build the log message for the given step and execution point (Start/End)
	 * 
	 * @param stepExecution
	 * @param executionPoint
	 * @return String log message
	 */
	private String getRemainingTimeMessage(StepExecution stepExecution) {
		final StringBuilder s = new StringBuilder();

		s.append("Job:").append(stepExecution.getJobExecution().getJobInstance().getJobName());
		s.append(", Step:").append(stepExecution.getStepName());

		long elapsedTimeMillis = 0L;
		elapsedTimeMillis += readCounter.stopWatch.getElapsedTime();
		elapsedTimeMillis += betweenReadProcessCounter.stopWatch.getElapsedTime();
		elapsedTimeMillis += processCounter.stopWatch.getElapsedTime();
		elapsedTimeMillis += betweenProcessWriteCounter.stopWatch.getElapsedTime();
		elapsedTimeMillis += writeCounter.stopWatch.getElapsedTime();
		elapsedTimeMillis += betweenWriteReadCounter.stopWatch.getElapsedTime();

		s.append(", Remaining Step Time:")
				.append(SimpleStopWatch.getElapsedTimeAsString(elapsedTimeMillis, stepStopWatch.getElapsedTime()));

		return s.toString();
	}

	private class ItemCounter {

		private StopWatch stopWatch = new SimpleStopWatch();

		private int itemsProcessed = 0;

		long getAverageTimePerItem() {
			return stopWatch.getElapsedTime() / ((itemsProcessed == 0) ? 1 : itemsProcessed);
		}
	}

}
