package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;

import static gov.va.cpss.model.ps.Constants.INPUT_DIRECTORY_KEY;
import static gov.va.cpss.model.ps.Constants.INPUT_RESOURCE_KEY;
import static gov.va.cpss.model.ps.Constants.MESSAGE_DIRECTORY_KEY;
import static gov.va.cpss.model.ps.Constants.MESSAGE_RESOURCE_KEY;
import static gov.va.cpss.model.ps.Constants.RECEIVED_ID_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;

import java.io.File;

import java.sql.Timestamp;

import java.util.ArrayList;
import java.util.List;

import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.apps.APSReceived;
import gov.va.cpss.model.updatestation.StationInfo;

import gov.va.cpss.service.LoadAPPSService;
import gov.va.cpss.service.Result;
import gov.va.cpss.service.StationInfoUtil;

/**
 * 
 * This is a scheduled batch job that will load APPS Payments Data, and save to
 * corresponding database tables.
 * 
 * Copyright HPE / VA
 * October 19, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 * 
 * This is a rewrite of the LoadAnnualVAMCPaymentDataJob, due to significant
 * requirements changes:
 * 
 *  1. Read multiple sites data in a single file, instead of one site one file;
 *  2. Continue processing (reading) when error occurs, instead of stopping;
 *  3. Write all errors to an error file, instead of partial errors to database;
 *  4. Format error messages with error codes based on the updated requirements;
 *  5. Validate data based on the updated requirements.
 * 
 * 
 * Copyright HPE / DXC / VA
 * March 27, 2016
 * 
 * @author Yiping Yao
 * @version 2.0.0
 *
 */
@SuppressWarnings("nls")
public class LoadAnnualVAMCPaymentDataJob extends AbstractProcessFileListJob
{
    private final static String DIVIDER = "; ";
    private final static String SPACER = " ";
    private final static int NUMBER_OF_SITES_PER_LINE = 1;

    protected APSReceived apsReceived;

	@Autowired
	private FlowJob LoadAnnualVAMCPaymentDataJobBatch;

    /**
     * The service used to manage job processing.
     */
    @Autowired
    protected LoadAPPSService loadAPPSService;

	@Autowired
	private String loadAPPSServerTargetDirectory;
	@Autowired
	private String loadAPPSServerArchiveDirectory;
	@Autowired
	private String loadAPPSServerErrorDirectory;
	
	@Override
	public String getDataDirectory()
	{
		return this.loadAPPSServerTargetDirectory;
	}
	
	@Override
	public String getArchiveSubDirectory()
	{
		return this.loadAPPSServerArchiveDirectory;
	}
	@Override
	public String getErrorSubDirectory()
	{
		return this.loadAPPSServerErrorDirectory;
	}

	@Override
	protected FlowJob getJob()
	{
		return this.LoadAnnualVAMCPaymentDataJobBatch;
	}

	/**
	 * Override the super class method, so that the Info Message can be built and appended.
	 */
	@Override
	protected boolean processList(final int batchRunId, FlowJob job, List<String> files)
	{
        boolean isSuccessful = super.processList(batchRunId, job, files);

        try
        {
            buildMessage();
        }
        catch (Exception ex)
        {
            StringBuilder errorBuilder = new StringBuilder("Unable to build info message.").append(LINE_FEED);

            errorBuilder.append(ex.getMessage()).append(LINE_FEED);

            jobLogger.error(errorBuilder.toString());

            // This is to avoid null point exception, since the super.appendErrorMessage uses the filename.
            this.filename = (this.filename == null) ? LINE_FEED + "Exception" : this.filename;

            super.appendErrorMessage(errorBuilder.toString());

            isSuccessful = false;
        }
        
        return isSuccessful;
    }

    private void buildMessage() throws Exception
    {
        StringBuilder msgBuilder = new StringBuilder();

        // Files
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("Total Files Received: ");
        msgBuilder.append(this.getFileCount()); // Read the total files received
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("Total Files Processed: ");
        msgBuilder.append(this.getSuccessfulFileCount()); // Read the total files processed
        msgBuilder.append(LINE_FEED);
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("Unprocessed Files: ");
        msgBuilder.append(getFileCount() - getSuccessfulFileCount()); // Read unprocessed files

        if (getFailedFileList() != null)
        {
            for (String fileName : getFailedFileList())
            {
                msgBuilder.append(LINE_FEED);
                msgBuilder.append(fileName);
            }
        }

        if (getErrorMessage() != null)
        {
            msgBuilder.append(getErrorMessage());
        }

        // Sites
        msgBuilder.append(LINE_FEED);
        msgBuilder.append(LINE_FEED);
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("Total VAMC Payment Files / Sites Received To Date: ");
        msgBuilder.append(getNumberOfFilesReceived()); // Read payment file received sites
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("VAMC Payment Files / Sites Remaining: ");
        msgBuilder.append(getNumberOfSitesRemaining()); // Read payment file remaining sites
        msgBuilder.append(LINE_FEED);
        msgBuilder.append("VAMC Payment Files Remaining from Sites:");

        if (getRemainingSites() != null)
        {
            int i = 0;

            for (String site : getRemainingSites())
            {
                if (i % NUMBER_OF_SITES_PER_LINE == 0)
                {
                    msgBuilder.append(LINE_FEED);
                }
                else
                {
                    msgBuilder.append(DIVIDER);
                }

                msgBuilder.append(site);
                i++;
            }
        }

        appendInfoMessage(msgBuilder.toString());
    }

	@Override
	protected boolean processFile(String file, int batchRunId, FlowJob job)
	{
	    boolean isSuccessful = true;

	    BatchRun batchRun = new BatchRun();

	    batchRun.setId(batchRunId);
	    batchRun.setStartDate(new Timestamp(this.loadAPPSService.getCurrentDate().getTime()));

	    Result<APSReceived> result = initializeJob(batchRun);

	    if (!result.isSuccessful())
	    {
	        super.appendErrorMessage(result.getMessage());
	        return false;
	    }

        this.apsReceived = result.get();

        JobParameters parameters = getParameters(this.apsReceived);

        JobExecution execution = executeJob(job, parameters);

        String message = this.loadAPPSService.endJob(execution, this.apsReceived);

        if (message != null)
        {
            super.appendErrorMessage(message);
            isSuccessful = false;
        }

        return isSuccessful;
	}

    /**
     * Initialize the job by calling the job service to obtain a database entry
     * representing the APSReceived for this run.
     * 
     * @return PSReceived object initialized by the JobService.
     */
    protected Result<APSReceived> initializeJob(final BatchRun batchRun)
    {
        // Use the job service to start the job.
        return this.loadAPPSService.startJob(batchRun, (new File(this.filename)).getName());
    }

    /**
     * Build parameters for the batch job based on the APSReceived object.
     * 
     * @param inAPSReceived
     *            The APSReceived object representing the database entry.
     * @return JobParameters object for this batch job run.
     */
    protected JobParameters getParameters(APSReceived inAPSReceived)
    {
        // Build job parameters for the input filename, directory, received primary key id, batch run id,
        // and error filename, directory.
        return new JobParametersBuilder().addString(INPUT_RESOURCE_KEY, this.filename)
                                         .addString(INPUT_DIRECTORY_KEY, getDataDirectory())
                                         .addString(MESSAGE_RESOURCE_KEY, this.filename)
                                         .addString(MESSAGE_DIRECTORY_KEY, getErrorDirectory())
                                         .addLong(RECEIVED_ID_KEY, new Long(inAPSReceived.getId()))
                                         .addLong(BATCH_RUN_ID_KEY, new Long(inAPSReceived.getBatchRunId()))
                                         .toJobParameters();
    }

    /**
     * 
     * Return the remaining sites that have not yet sent the payment files,
     * in the form of: Site Id (Station / Facility Number) Site Name
     *   
     */
    private List<String> getRemainingSites()
    {
        return buildSites(this.loadAPPSService.getRemainingStations());
    }

    /**
     * 
     * Return the sites that have sent the payment files,
     * in the form of: Site Id (Station / Facility Number) Site Name
     *   
     */
    private List<String> getPaymentReceivedSites()
    {
        return buildSites(this.loadAPPSService.getPaymentRecievedStations());
    }

    /**
     * Build site with Site ID and Site Name from Station Info.
     */
    private static List<String> buildSites(List<StationInfo> stations)
    {
        List<String> sites = new ArrayList<>();

        if (stations != null && !stations.isEmpty())
        {
            for (StationInfo station : stations)
            {
                String site = station.getStationNum() + SPACER + station.getFacilityDesc();

                sites.add(site);
            }
        }

        return sites;
    }

    /**
     * 
     * Return the number of payment files / sites received to date.
     * 
     */
    private int getNumberOfFilesReceived()
    {
        return (getPaymentReceivedSites() == null || getPaymentReceivedSites().isEmpty()) ? 0 : getPaymentReceivedSites().size();
    }

    /**
     * 
     * Return the number of remaining sites that have not yet sent the payment files.
     * 
     */
    private int getNumberOfSitesRemaining()
    {
        return (getRemainingSites() == null || getRemainingSites().isEmpty()) ? StationInfoUtil.getTotalNumberOfStations() : getRemainingSites().size(); 
    }
}