package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_AVAILABLE_STATISTICS_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_DETECTED_PATIENTS_ERROR_KEY;
import static gov.va.cpss.job.cbs.CbsProcessingConstants.CBS_GENERATED_STATISTICS_KEY;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;

public class GenerateCBSJob extends CbssBaseJob {
	
	@Autowired
	private FlowJob GenerateCBSJobBatch;

	private String availableStatistics = null;

	private String generatedStatistics = null;

	private String detectedPatientsErrorMessage = null;

	@Override
	protected FlowJob getJob() {
		return GenerateCBSJobBatch;
	}
	
	@Override
	protected boolean runJob(BatchRun bR, FlowJob job) {

		boolean successful = false;

		initializeJobStatistics();

		JobExecution execution = executeJob(job, getParameters(bR.getId()));

		successful = processJobExecution(execution);

		return successful;
	}

	/**
	 * Initialize statistics for job run.
	 */
	private void initializeJobStatistics() {
		availableStatistics = null;
		generatedStatistics = null;
		detectedPatientsErrorMessage = null;
	}

	/**
	 * Build parameters for the batch job.
	 * 
	 * @return JobParameters object for this batch job run.
	 */
	protected JobParameters getParameters(final int batchRunId) {

		// Build job parameters for the batch run.
		return new JobParametersBuilder().addLong(BATCH_RUN_ID_KEY, new Long(batchRunId)).toJobParameters();
	}

	/**
	 * Process the batch job execution and verify the exit status.
	 * 
	 * @param execution
	 *            The job execution.
	 * @return Boolean flag indicating if processed successfully or not.
	 */
	private boolean processJobExecution(JobExecution execution) {

		boolean jobSuccess = false;

		if (execution == null) {

			jobLogger.error("Job ended with null execution");

		} else if (!execution.getExitStatus().equals(ExitStatus.COMPLETED)) {

			if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY)) {
				jobLogger
						.error("Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY));
			} else {
				jobLogger.error("Job ended with unknown error: " + execution.getExitStatus());
			}

		} else {

			// According to the execution context the job should be successful.
			jobSuccess = true;
		}

		// Perform one last check to make sure the expected results were
		// captured.
		// Pass in current state of job success. The final job success result
		// will be returned.
		jobSuccess = captureJobExecutionResults(execution, jobSuccess);

		return jobSuccess;
	}

	/**
	 * Capture the batch job results from the job execution.
	 * 
	 * @param execution
	 *            The job execution.
	 * @return Boolean indicating if results were processed successfully or not.
	 */
	private boolean captureJobExecutionResults(JobExecution execution, final boolean currentSuccess) {

		boolean completeSuccess = currentSuccess;

		// How many statements are available?
		if (execution.getExecutionContext().containsKey(CBS_AVAILABLE_STATISTICS_KEY)) {
			availableStatistics = execution.getExecutionContext().getString(CBS_AVAILABLE_STATISTICS_KEY);
		} else {
			jobLogger.error("Unable to obtain key from process results: " + CBS_AVAILABLE_STATISTICS_KEY);
			completeSuccess = false;
		}

		// How many statements were generated?
		if (execution.getExecutionContext().containsKey(CBS_GENERATED_STATISTICS_KEY)) {
			generatedStatistics = execution.getExecutionContext().getString(CBS_GENERATED_STATISTICS_KEY);
		} else {
			jobLogger.error("Unable to obtain key from process results: " + CBS_GENERATED_STATISTICS_KEY);
			completeSuccess = false;
		}

		// Were there any data errors?
		// This is only reported in the job context if such errors are detected.
		if (execution.getExecutionContext().containsKey(CBS_DETECTED_PATIENTS_ERROR_KEY)) {
			detectedPatientsErrorMessage = execution.getExecutionContext().getString(CBS_DETECTED_PATIENTS_ERROR_KEY);
		}

		// Build informational message.
		buildInfoMessage();

		return completeSuccess;
	}

	/**
	 * Build this job specific informational message.
	 */
	private void buildInfoMessage() {

		StringBuffer strBuff = new StringBuffer();
		strBuff.append("\n");
		strBuff.append(availableStatistics);
		strBuff.append("\n");
		strBuff.append(generatedStatistics);

		if ((detectedPatientsErrorMessage != null) && !detectedPatientsErrorMessage.isEmpty()) {
			strBuff.append("\n\n");
			strBuff.append(detectedPatientsErrorMessage);
		}

		appendInfoMessage(strBuff.toString());
	}
}