package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;

import static gov.va.cpss.model.ps.Constants.BATCH_PROCESS_ID_KEY;
import static gov.va.cpss.model.ps.Constants.AVAILABLE_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.GENERATED_CS_STATISTICS_KEY;
import static gov.va.cpss.model.ps.Constants.DETECTED_ERROR_KEY;
import static gov.va.cpss.model.ps.Constants.LINE_FEED;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.JobParametersBuilder;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchRunProcess;

import gov.va.cpss.service.GenerateAPPSService;
import gov.va.cpss.service.Result;

/**
 * 
 * This is a scheduled batch job that will generate / consolidate APPS Payments Data,
 * and save to corresponding database tables.
 * 
 * Copyright HPE / VA
 * December 2, 2016
 * January 17, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 */
@SuppressWarnings("nls")
public class GenerateAPPSDataJob extends CbssBaseJob
{
    private final static String DIVIDER_LINE = "*********************************************";

    @Autowired
    private FlowJob GenerateAPPSDataJobBatch;

    /**
     * The service used to manage job processing.
     */
    @Autowired
    protected GenerateAPPSService generateAPPSService;

    private String availableStatistics = null;
    private String generatedStatistics = null;
    private String detectedErrorMessage = null;

    @Override
    protected FlowJob getJob()
    {
        return this.GenerateAPPSDataJobBatch;
    }
    
    @Override
    protected boolean runJob(BatchRun batchRun, FlowJob job)
    {
        String message = null;

        Result<BatchRunProcess> result = initializeJob(batchRun);

        if (!result.isSuccessful())
        {
            super.appendErrorMessage(result.getMessage());
            return false;
        }

        BatchRunProcess process = result.get();

        if (process != null && process.getId() > 0)
        {
            JobExecution execution = executeJob(job, getParameters(process));

            this.generateAPPSService.endJob(execution, process);

            message = processJobExecution(execution);
        }

        if (message != null)
        {
            super.appendErrorMessage(message);
            return false;
        }

        return true;
    }

    /**
     * Initialize the job process.
     */
    protected Result<BatchRunProcess> initializeJob(BatchRun batchRun)
    {
        initializeJobStatistics();

        return this.generateAPPSService.startJob(batchRun, "Generate APPS"); 
    }

    /**
     * Initialize statistics for job run.
     */
    private void initializeJobStatistics()
    {
        this.availableStatistics = null;
        this.generatedStatistics = null;
        this.detectedErrorMessage = null;
    }

    /**
     * Build parameters for the batch job.
     * 
     * @return JobParameters object for this batch job run.
     */
    @SuppressWarnings("static-method")
    protected JobParameters getParameters(final BatchRunProcess process)
    {
        // Build job parameters for the batch run.
        return new JobParametersBuilder().addLong(BATCH_RUN_ID_KEY, new Long(process.getBatchRunId()))
                                         .addLong(BATCH_PROCESS_ID_KEY, new Long(process.getId()))
                                         .toJobParameters();
    }

    /**
     * Process the batch job execution and verify the exit status.
     * 
     * @param execution
     *            The job execution.
     * @return Boolean flag indicating if processed successfully or not.
     */
    protected String processJobExecution(JobExecution execution)
    {
        String message = null;

        if (execution == null)
        {
            message = "Job ended with null execution";
            jobLogger.error(message);
        }
        else if (!execution.getExitStatus().equals(ExitStatus.COMPLETED))
        {
            if (execution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
            {
                message = "Job ended with failure: " + execution.getExecutionContext().getString(JOB_FAILURE_KEY);
                jobLogger.error(message);
            }
            else
            {
                message = "Job ended with unknown error: " + execution.getExitStatus();
                jobLogger.error(message);
            }
        }

        // Perform one last check to make sure the expected results were captured.
        // The final job success result will be returned.
        String tempMessage = captureJobExecutionResults(execution);

        if (tempMessage != null)
        {
            message = (message == null) ? tempMessage : message + LINE_FEED + tempMessage;
        }

        return message;
    }

    /**
     * Capture the batch job results from the job execution.
     * 
     * @param execution
     *            The job execution.
     * @return String indicating if results were processed successfully or not.
     */
    private String captureJobExecutionResults(JobExecution execution)
    {
        String message = null;

        // How many statements are available?
        if (execution.getExecutionContext().containsKey(AVAILABLE_CS_STATISTICS_KEY))
        {
            this.availableStatistics = execution.getExecutionContext().getString(AVAILABLE_CS_STATISTICS_KEY);
        }
        else
        {
            message = "Unable to obtain key from process results: " + AVAILABLE_CS_STATISTICS_KEY;
            jobLogger.error(message);
        }

        // How many statements were generated?
        if (execution.getExecutionContext().containsKey(GENERATED_CS_STATISTICS_KEY))
        {
            this.generatedStatistics = execution.getExecutionContext().getString(GENERATED_CS_STATISTICS_KEY);
        }
        else
        {
            message += LINE_FEED + "Unable to obtain key from process results: " + GENERATED_CS_STATISTICS_KEY;
            jobLogger.error(message);
        }

        // Were there any data errors?
        // This is only reported in the job context if such errors are detected.
        if (execution.getExecutionContext().containsKey(DETECTED_ERROR_KEY))
        {
            this.detectedErrorMessage = execution.getExecutionContext().getString(DETECTED_ERROR_KEY);
        }

        // Build informational message.
        buildInfoMessage();

        return message;
    }

    /**
     * Build this job specific informational message.
     */
    private void buildInfoMessage()
    {
        StringBuilder strBuilder = new StringBuilder();
        strBuilder.append(LINE_FEED);
        strBuilder.append("Total APPS Patient Records Available / Read / Loaded: ");
        strBuilder.append(this.availableStatistics);
        strBuilder.append(LINE_FEED);
        strBuilder.append("Total APPS Statements Generated: ");
        strBuilder.append(this.generatedStatistics);

        if ((this.detectedErrorMessage != null) && !this.detectedErrorMessage.isEmpty())
        {
            strBuilder.append(LINE_FEED);
            strBuilder.append(LINE_FEED);
            strBuilder.append(this.detectedErrorMessage);
        }

        appendInfoMessage(strBuilder.toString());
    }

    @Override
    protected String buildEmailCustomInfo()
	{
		StringBuilder strBuilder = new StringBuilder();

        strBuilder.append(LINE_FEED);
        strBuilder.append(LINE_FEED);
        strBuilder.append(DIVIDER_LINE);
        strBuilder.append(LINE_FEED);
		strBuilder.append(super.buildEmailCustomInfo());
		strBuilder.append(LINE_FEED);
        strBuilder.append(LINE_FEED);
        strBuilder.append(DIVIDER_LINE);
        strBuilder.append(LINE_FEED);
        strBuilder.append("Please don't reply directly to this automatically generated email message.");
        strBuilder.append(LINE_FEED);

		return strBuilder.toString();
	}

}