package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.FTP_OPEN_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.READ_FAILURE_STATUS;

import java.io.InputStream;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeoutException;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;
import org.springframework.batch.item.file.FlatFileParseException;
import org.springframework.batch.item.file.LineMapper;
import org.springframework.batch.item.file.ResourceAwareItemReaderItemStream;
import org.springframework.batch.item.support.AbstractItemCountingItemStreamItemReader;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

import gov.va.cpss.service.SftpService;
import gov.va.cpss.service.SftpStreamSession;

/**
 * Implementation of ItemCountingItemStreamItemReader used by a batch job to
 * read data over an sftp file stream.
 * 
 * @author DNS 
 */
public abstract class CbssSftpItemReader<T> extends AbstractItemCountingItemStreamItemReader<T>
		implements ResourceAwareItemReaderItemStream<T>, InitializingBean, StepExecutionListener {

	private final int DEFAULT_READ_ATTEMPT_COUNT = 30;

	protected static final Logger readerLogger = Logger.getLogger(CbssSftpItemReader.class.getCanonicalName());

	protected boolean opened = false;

	private SftpService sftpService;

	protected JobExecution jobExecution;

	private int lineCount = 0;

	protected Resource resource;

	private String directory;

	private LineMapper<T> lineMapper;

	protected BlockingQueue<String> dataQueue = new LinkedBlockingQueue<>();
	
	private SftpStreamSession sftpStreamSession;

	protected CbssStreamToQueueThread queueBuilderThread;

	public CbssSftpItemReader() {
		setName(ClassUtils.getShortName(CbssSftpItemReader.class));
	}

	public SftpService getSftpService() {
		return sftpService;
	}

	public void setSftpService(SftpService sftpService) {
		this.sftpService = sftpService;
	}

	public LineMapper<T> getLineMapper() {
		return lineMapper;
	}

	public void setLineMapper(LineMapper<T> lineMapper) {
		this.lineMapper = lineMapper;
	}

	public String getDirectory() {
		return directory;
	}

	public void setDirectory(String directory) {
		this.directory = directory;
	}

	@Override
	protected void doOpen() throws Exception {

		opened = false;

		try {

			if (resource.getFilename() != null) {

				// Get the size of the file.
				final Long expectedByteCount = sftpService.ftpGetFileSizeInDirectory(resource.getFilename(), directory);

				if (expectedByteCount != null) {

					readerLogger.info("Attempting to read file from ftp server: " + resource.getFilename());

					// Open the file download stream session.
					sftpStreamSession = sftpService.openFileStream(resource.getFilename(), directory);

					if (sftpStreamSession != null) {

						// Start thread to populate queue.
						queueBuilderThread = getQueueBuilderThread(expectedByteCount, sftpStreamSession.getInputStream(), dataQueue);
						if (queueBuilderThread != null) {
							queueBuilderThread.start();
							opened = true;
						} else {
							setFailureStatusAndMessage(FTP_OPEN_ERROR_STATUS,
									"Unable to open ftp connection because failed to obtain data stream processing thread");
						}

					} else {
						setFailureStatusAndMessage(FTP_OPEN_ERROR_STATUS,
								"Unable to open ftp connection because failed to obtain data stream");
					}

				} else {
					setFailureStatusAndMessage(FTP_OPEN_ERROR_STATUS,
							"Unable to open ftp connection because failed file size request");
				}
			}

		} catch (Exception e) {
			StringBuilder error = new StringBuilder();
			error.append("Unable to open ftp connection for read because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			setFailureStatusAndMessage(FTP_OPEN_ERROR_STATUS, error.toString());
		}
	}

	@Override
	public T read() throws Exception, UnexpectedInputException, ParseException {
		try {

			return super.read();

		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Unable to read item because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());
			if ((e.getCause() != null) && (e.getCause().getMessage() != null)) {
				error.append("\nCause: ");
				error.append(e.getCause().getMessage().trim());
			}

			setFailureStatusAndMessage(READ_FAILURE_STATUS, error.toString());
		}
		// Null response will cause step to end.
		return null;
	}

	/**
	 * Subclass must implement method for the appropriate stream to queue
	 * builder thread.
	 * 
	 * @return The queue builder thread.
	 */
	protected abstract CbssStreamToQueueThread getQueueBuilderThread(long expectedByteCount, InputStream in,
			BlockingQueue<String> outputQueue);

	@Override
	public void beforeStep(StepExecution stepExecution) {
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set exit status if a failure
		// during read processing.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		// Do not do anything special here.
		return null;
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	protected void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);

		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		readerLogger.error("Read failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		readerLogger.error("Read failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}

	@Override
	protected T doRead() throws Exception {

		// If not opened then return null which will effectively stop
		// processing.
		if (!opened) {
			return null;
		}

		// Local flag to retry read until failure or get line.
		boolean processingStream = true;

		// Local counter to retry so many times before declaring a read failure.
		int readAttemptCounter = 0;
		do {

			try {
				// Increment the attempt counter to make sure we do not get
				// stuck infinite loop.
				++readAttemptCounter;

				// Obtain the next message.
				String line = readLine();

				if (line == null) {

					// If thread is not alive or queue is empty then quit
					// processing.
					// Otherwise, will attempt to read again.
					if (!queueBuilderThread.isAlive() && dataQueue.isEmpty()) {
						processingStream = false;
					} else {
						readerLogger.info("Waiting for queue...");
						Thread.sleep(1000);
					}

				} else {

					try {
						// Convert the line to a mapped object.
						return lineMapper.mapLine(line, lineCount);
					} catch (Exception ex) {
						throw new FlatFileParseException("Parsing error at line: " + lineCount + " in resource=["
								+ resource.getDescription() + "], input=[" + line + "]", ex, line, lineCount);
					}
				}

			} catch (InterruptedException e) {
				readerLogger.error("Queueing thread not dead and will attempt retry: " + e.getMessage());
			}

			// If attempted too many read attempts then declare failure.
			if (readAttemptCounter > DEFAULT_READ_ATTEMPT_COUNT) {
				final String error = "Failed to read from data queue within allocated number of retries ("
						+ DEFAULT_READ_ATTEMPT_COUNT + ")";
				readerLogger.error(error);
				processingStream = false;
				throw new TimeoutException(error);
			}

		} while (processingStream);

		return null;
	}

	/**
	 * Read the line from the data queue. If an error is encountered it should
	 * ripple up and be caught by read method.
	 * 
	 * @return The next row of the file.
	 * @throws Exception
	 */
	protected String readLine() throws Exception {

		if (!dataQueue.isEmpty()) {
			lineCount++;
			return dataQueue.take();
		}

		return null;
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		Assert.notNull(lineMapper, "LineMapper is required");
	}

	@Override
	public void setResource(Resource resource) {
		this.resource = resource;
	}

	@Override
	protected void doClose() throws Exception {
		
		// Be sure to reset the line count.
		lineCount = 0;

		// Stop the queue reader thread from processing.
		// Wait up to 10 seconds for the thread to stop processing.
		if (queueBuilderThread != null) {
			queueBuilderThread.interrupt();
			queueBuilderThread.join(10000);
		}
		
		// Close the session and the input stream.
		if (sftpStreamSession != null) {
			sftpStreamSession.close();
		}
	}

}
