package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.FILE_OPEN_ERROR_STATUS;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.READ_FAILURE_STATUS;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ParseException;
import org.springframework.batch.item.UnexpectedInputException;
import org.springframework.batch.item.file.FlatFileItemReader;

public abstract class CbssFlatFileItemReader<T> extends FlatFileItemReader<T> implements StepExecutionListener {
	
	private JobExecution jobExecution;

	private final Logger readerLogger;

	public CbssFlatFileItemReader() {
		readerLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	@Override
	protected void doOpen() throws Exception {
		try {
			super.doOpen();
		} catch (Exception e) {
			StringBuilder error = new StringBuilder();
			error.append("Unable to open for read because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());

			setFailureStatusAndMessage(FILE_OPEN_ERROR_STATUS, error.toString());
		}
	}
	
	@Override
	protected T doRead() throws Exception {
		try {
			return super.doRead();
		} catch(Exception e) {
			StringBuilder error = new StringBuilder();
			error.append("Unable to read item because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			setFailureStatusAndMessage(READ_FAILURE_STATUS, error.toString());
		}
		return null;
	}

	@Override
	public T read() throws Exception, UnexpectedInputException, ParseException {
		try {
			return super.read();
		} catch (Exception e) {

			StringBuilder error = new StringBuilder();
			error.append("Unable to read item because of ");
			error.append(e.getClass().getSimpleName());
			error.append("\nMessage: ");
			error.append(e.getMessage());
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());

			setFailureStatusAndMessage(READ_FAILURE_STATUS, error.toString());
		}
		// Null response will cause step to end.
		return null;
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		// Save the job execution at the beginning of the step.
		// The execution context will be used to set exit status if a failure
		// during read processing.
		jobExecution = stepExecution.getJobExecution();
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		// Do not do anything special here.
		return null;
	}

	/**
	 * Set the failure and message in the job execution context.
	 */
	private void setFailureStatusAndMessage(final String status, final String message) {
		// Set job failure.
		setFailureStatus(status);

		// Set job failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 */
	private void setFailureStatus(final String status) {
		// Log job failure status.
		readerLogger.error("Read failed with status: " + status);

		// Set job failure.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 */
	private void setFailureMessage(final String message) {
		// Log job failure message.
		readerLogger.error("Read failure message: " + message);

		// Set job failure message.
		jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}
}
