package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;

import org.apache.log4j.Logger;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.step.item.BatchRetryTemplate;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;
import org.springframework.retry.RecoveryCallback;
import org.springframework.retry.RetryCallback;
import org.springframework.retry.RetryContext;
import org.springframework.retry.RetryException;

/**
 * Abstract base class that implements logic for a tasklet to retry execution.
 * 
 * @author DNS 
 */
public abstract class CbssBaseRetryTasklet implements Tasklet, StepExecutionListener {

	private static final String RETRY_FAILURE_STATUS = "RETRY ATTEMPTS EXHAUSTED";
	private static final String RETRY_FAILURE_MESSAGE = "Processing could not continue due to retry constraints";

	protected final Logger taskletLogger;

	/*
	 * This flag is used to indicate if the processing was successful and no
	 * longer needs to retry.
	 */
	private boolean successful = false;

	/*
	 * The retry template that has the retry and backoff policy configurations.
	 */
	private BatchRetryTemplate batchRetryTemplate;

	public CbssBaseRetryTasklet() {
		taskletLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public BatchRetryTemplate getBatchRetryTemplate() {
		return batchRetryTemplate;
	}

	public void setBatchRetryTemplate(BatchRetryTemplate batchRetryTemplate) {
		this.batchRetryTemplate = batchRetryTemplate;
	}

	@Override
	public RepeatStatus execute(StepContribution contribution, ChunkContext chunkContext) throws Exception {

		return batchRetryTemplate.execute(new RetryCallback<RepeatStatus, RetryException>() {

			public RepeatStatus doWithRetry(RetryContext context) throws RetryException {

				// Call custom logic implemented by subclass.
				if (executeLogic()) {
					successful = true;
					return RepeatStatus.FINISHED;
				}

				taskletLogger.warn("Tasklet could not continue, attempting retry...");
				throw new RetryException("Tasklet could not continue, attempting retry...");
			}

		}, new RecoveryCallback<RepeatStatus>() {

			@Override
			public RepeatStatus recover(RetryContext context) throws Exception {

				taskletLogger.error("Tasklet could not start and exhausted all retry attempts");
				return RepeatStatus.FINISHED;
			}

		});
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		// Initialize.
		successful = false;
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {

		if (successful) {
			return ExitStatus.COMPLETED;
		}

		stepExecution.getJobExecution().getExecutionContext().putString(JOB_FAILURE_KEY, RETRY_FAILURE_STATUS);
		stepExecution.getJobExecution().getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, RETRY_FAILURE_MESSAGE);
		return ExitStatus.FAILED;
	}

	/**
	 * Custom logic provided by implementing class to return true of tasklet
	 * completed successfully and no longer needs to attempt retry.
	 * 
	 * @return Boolean flag indicating if the logic was executed successfully.
	 */
	abstract protected boolean executeLogic();

}
