package gov.va.cpss.job;

import java.util.List;

import org.apache.log4j.Logger;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.JobParameters;
import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.batch.core.launch.JobLauncher;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.quartz.QuartzJobBean;

import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.performance.PerformanceAnalyzer;
import gov.va.cpss.service.BatchService;
import gov.va.cpss.service.EmailService;

/**
 * Base job class for CBSS quartz jobs. All custom jobs should extend this class
 * and implement the runJob method.
 * 
 * @author DNS  
 */
public abstract class CbssBaseJob extends QuartzJobBean {

	protected static final Logger jobLogger = Logger.getLogger(CbssBaseJob.class.getCanonicalName());
	
	private static final int MAX_ERROR_LENGTH = 2048;

	/**
	 * The batch job name that is populated from the database at scheduler
	 * initialization.
	 */
	protected String name;

	/**
	 * The service used to manage batch issues.
	 */
	@Autowired
	protected BatchService batchService;

	/**
	 * Spring Batch JobLauncher for starting FlowJobs
	 */
	@Autowired
	protected JobLauncher jobLauncher;

	/**
	 * The service used to send email.
	 */
	@Autowired
	protected EmailService emailService;

	/**
	 * Report memory usage.
	 */
	@Autowired
	protected PerformanceAnalyzer memoryAnalyzer;

	/*
	 * The informational portion of the message for the batch run.
	 */
	private String infoMessage;

	/*
	 * The error portion of the message for the batch run.
	 */
	private String errorMessage;

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getInfoMessage() {
		return infoMessage;
	}

	public String getErrorMessage() {
		return errorMessage;
	}

	public void resetMessages() {
		infoMessage = null;
		errorMessage = null;
	}

	/**
	 * The entry point for job execution.
	 */
	@Override
	protected void executeInternal(JobExecutionContext ctx) throws JobExecutionException {

		// Ensure no other quartz job is running.
		if (notAlreadyRunning(ctx)) {

			// Report Memory Before Processing
			memoryAnalyzer.report("at " + name + " Start");

			// Reset the message.
			resetMessages();

			// Get the batch job.
			BatchJob bj = batchService.getBatchJobByName(name);

			// Start a batch run.
			BatchRun bR = batchService.startRun(bj.getId());

			if (bR != null) {

				jobLogger.info("Job " + name + " Started: " + bR.getStartDate());

				if (getJob() == null) {
					jobLogger.error("Job batch is null");
					errorRun(bR);
				} else {
					try {
						final boolean runSuccess = runJob(bR, getJob());
						
						if (runSuccess) {
							completeRun(bR);
						} else {
							errorRun(bR);
						}
					} catch (Exception e) {
						final String errorMessage = "Job " + name + " encountered an error: " + e.getMessage();
						jobLogger.error(errorMessage, e);
						appendErrorMessage(errorMessage);
						errorRun(bR);
					}
				}

				emailService.cpssReadyToSendEmail(bj.getEmail(), buildEmailSubject(bj), buildEmailMessage(bj, bR));

			} else {
				final String errorMessage = "Problem Starting Job " + name;
				jobLogger.error(errorMessage);
				emailService.cpssReadyToSendEmail(bj.getEmail(), buildEmailSubject(bj), errorMessage);
			}

			// Report Memory After Processing
			memoryAnalyzer.report("at " + name + " End");
		}
	}

	/**
	 * Ensure the quartz job is not already running.
	 * 
	 * @param ctx
	 *            The job execution context for the current job.
	 * @return Boolean flag indicating if the job is not already running.
	 */
	private boolean notAlreadyRunning(JobExecutionContext ctx) {

		boolean running = false;

		try {
			Scheduler scheduler = ctx.getScheduler();
			List<JobExecutionContext> jobs = scheduler.getCurrentlyExecutingJobs();
			for (JobExecutionContext job : jobs) {
				// We are only concerned where the job detail is the same as
				// this job.
				if (job.getJobDetail().equals(ctx.getJobDetail())) {
					// If a different trigger then it is obviously a duplicate
					// so do not allow to run.
					// If it is same trigger but different fire instance ID then
					// it is also a duplicate so do not allow to run.
					if (!job.getTrigger().equals(ctx.getTrigger())) {
						running = true;
						jobLogger.warn(
								"Attempted to start job but a different context is already running with a different trigger");
						break;
					} else if (!job.getFireInstanceId().equals(ctx.getFireInstanceId())) {
						running = true;
						jobLogger.warn(
								"Attempted to start job but a different context is already running with the same trigger");
						break;
					}
				}
			}
		} catch (SchedulerException e) {
			// If scheduler has a problem then assume fatal error and do not run
			// job.
			jobLogger.error("Could not check if job already running: " + e.getMessage());
			running = true;
		}

		return !running;
	}

	/**
	 * Complete and end a successful job run.
	 * 
	 * @param bR
	 *            The batch run database entry.
	 */
	protected void completeRun(BatchRun bR) {

		setMessage(bR);

		if (batchService.completeRun(bR)) {
			jobLogger.info("Job " + name + " Ended: " + bR.getEndDate());
		} else {
			jobLogger.error("Problem Ending Job " + name);
		}
	}

	/**
	 * Error and end a job run.
	 * 
	 * @param bR
	 *            The batch run database entry.
	 */
	protected void errorRun(BatchRun bR) {

		setMessage(bR);

		if (batchService.errorRun(bR)) {
			jobLogger.error("Job " + name + " Completed with Error: " + bR.getEndDate());
		} else {
			jobLogger.error("Problem Ending Job " + name + " with Error");
		}
	}
	
	/**
	 * Get the batch job associated with this quartz job.
	 * @return The associated job.
	 */
	protected abstract FlowJob getJob();

	/**
	 * Entry method to run the batch job. Subclasses should implement this
	 * method with custom business logic.
	 * 
	 * @param job
	 *            The batch job to run.
	 * @param bR
	 *            The batch run for this job.
	 * @return Boolean value that indicates if the batch job ran successfully or
	 *         not.
	 */
	protected abstract boolean runJob(BatchRun bR, FlowJob job);

	/**
	 * Execute the specified batch job using the specified parameters.
	 * 
	 * @param job
	 *            The batch job.
	 * @param parameters
	 *            The parameters to use for job execution.
	 * @return Boolean value indicating success or failure of the job run.
	 */
	protected JobExecution executeJob(FlowJob job, JobParameters parameters) {

		JobExecution execution = null;

		try {

			execution = jobLauncher.run(job, parameters);

			if (!execution.getExitStatus().equals(ExitStatus.COMPLETED)) {
				if (!execution.getExitStatus().getExitDescription().isEmpty()) {
					appendErrorMessage(execution.getExitStatus().getExitDescription());
				} else {
					appendErrorMessage(execution.getExitStatus().getExitCode().toString());
				}
			}

		} catch (Exception e) {

			logExecuteJobException("Caught exception during job execution: " + e.getMessage());

			if (execution == null) {
				appendErrorMessage("Job ended with null execution");
			}
		}

		return execution;
	}

	/**
	 * Append the informational message. This allows capture of multiple
	 * informational messages over the course of multiple job runs.
	 */
	protected void appendInfoMessage(final String message) {
		StringBuilder infoBuilder = new StringBuilder();
		if ((infoMessage != null) && !infoMessage.isEmpty()) {
			infoBuilder.append(infoMessage);
			infoBuilder.append("\n");
		}
		infoBuilder.append(message);
		infoMessage = infoBuilder.toString();
	}

	/**
	 * Append the error message. This allows capture of multiple errors over the
	 * course of multiple job runs.
	 */
	protected void appendErrorMessage(final String message) {
		StringBuilder errorBuilder = new StringBuilder();
		if ((errorMessage != null) && !errorMessage.isEmpty()) {
			errorBuilder.append(errorMessage);
			errorBuilder.append("\n");
		} else {
			errorBuilder.append("\n\nError Message:\n");
		}
		errorBuilder.append(message);
		errorMessage = errorBuilder.toString();
	}

	/**
	 * Log exception in executeJob. Moved to separate method so subclasses can
	 * override.
	 */
	protected void logExecuteJobException(final String message) {
		jobLogger.error(message);
	}

	/**
	 * Set the batch run message to save in the database.
	 */
	private void setMessage(BatchRun bR) {
		StringBuilder messageBuilder = new StringBuilder();

		if ((infoMessage != null) && !infoMessage.isEmpty()) {
			messageBuilder.append(infoMessage);
		}

		if ((errorMessage != null) && !errorMessage.isEmpty()) {
			messageBuilder.append(errorMessage);
		}

		String runMessage = messageBuilder.toString();

		if (runMessage.length() > MAX_ERROR_LENGTH) {
			jobLogger.warn("Error message will be truncated in the database");
		}

		// If necessary, truncate error message to max size allowed by table.
		bR.setMessage(runMessage.substring(0, Math.min(MAX_ERROR_LENGTH, runMessage.length())));
	}

	/**
	 * Build an email subject.
	 * 
	 * @return
	 */
	protected String buildEmailSubject(BatchJob batchJob) {
		return batchJob.getDescription();
	}

	/**
	 * Build an email message that communicates information based on the
	 * specific type of job.
	 */
	protected String buildEmailMessage(BatchJob batchJob, BatchRun batchRun) {
		StringBuffer strBuff = buildEmailCommonInfo(batchJob, batchRun);
		strBuff.append(buildEmailCustomInfo());
		strBuff.append(buildEmailErrorInfo());
		return strBuff.toString();
	}

	/**
	 * Build the first portion of an email with information that every job has
	 * in common.
	 */
	protected StringBuffer buildEmailCommonInfo(BatchJob batchJob, BatchRun batchRun) {
		StringBuffer strBuff = new StringBuffer();
		strBuff.append("\nBatch Process Name: ");
		strBuff.append(batchJob.getDescription()); // Read the Job class running
		strBuff.append("\nStart: ");
		strBuff.append(batchRun.getServerStartDateString()); // Read the start
																// date
		strBuff.append("\nEnd: ");
		strBuff.append(batchRun.getServerEndDateString()); // Read the end date
		strBuff.append("\nStatus: ");
		strBuff.append(batchRun.getBatchStatus().getDescription()); // Read the
																	// status
																	// description
		return strBuff;
	}

	/**
	 * Build the error portion of an email.
	 */
	protected String buildEmailErrorInfo() {
		if ((errorMessage != null) && !errorMessage.isEmpty()) {
			return errorMessage;
		}
		return "";
	}

	/**
	 * Build the second portion of an email that contains information specific
	 * to the derived job class.
	 */
	protected String buildEmailCustomInfo() {
		if ((infoMessage != null) && !infoMessage.isEmpty()) {
			return infoMessage;
		}
		return "";
	}
}
