package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.RESULTS_SIZE_KEY;
import static gov.va.cpss.model.ps.Constants.TOTAL_READ_COUNT_KEY;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.database.AbstractPagingItemReader;

/**
 * 
 * This is an abstract item reader with pagination.
 * The purpose of this class is to use CBS service / DAO layer
 * to retrieve data one page at a time.
 * 
 * Copyright HPE / VA
 * January 25, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 */
@SuppressWarnings("nls")
public abstract class CBSSPagingItemReader<T> extends AbstractPagingItemReader<T> implements StepExecutionListener
{
    protected JobExecution jobExecution;

    protected Map<String, Object> parameterValues;

    public JobExecution getJobExecution()
    {
		return this.jobExecution;
	}

	/**
     * The parameter values to apply to the reader (map of name:value).
     *
     * @param inParameterValues the parameter values to set
     */
    public void setParameterValues(Map<String, Object> inParameterValues)
    {
        this.parameterValues = inParameterValues;
    }

    /**
     * Before reading, capture the job execution.
     * 
     * @param stepExecution
     */
    @Override
    public void beforeStep(StepExecution stepExecution)
    {
        this.logger.info("Before Step Execution ...");

        // Get the job execution at the beginning of the step.
        // And put the page size on the Execution Context for Processor / Writer. 
        this.jobExecution = stepExecution.getJobExecution();

        // Initialize the counters.
        this.jobExecution.getExecutionContext().putInt(RESULTS_SIZE_KEY, 0);
        this.jobExecution.getExecutionContext().putLong(TOTAL_READ_COUNT_KEY, 0);
    }

    /**
     * After processing check for unexpected conditions that suggest an error.
     * 
     * @param stepExecution
     * @return 
     */
    @Override
    public ExitStatus afterStep(StepExecution stepExecution)
    {
        this.logger.info("After Step Execution.");

        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            return ExitStatus.FAILED;
        }

        // Reset the counters.
        this.jobExecution.getExecutionContext().putInt(RESULTS_SIZE_KEY, 0);

        return ExitStatus.COMPLETED;
    }

    @Override
    protected void doReadPage()
    {
        if (this.results == null)
        {
            this.results = new CopyOnWriteArrayList<>();
        }
        else
        {
            this.results.clear();
        }

        // The AbstractPagingItemReader starts the page at zero (0).
        this.results.addAll(executePageRead(getPage() + 1, getPageSize(), this.parameterValues));

        // Read results size
        this.jobExecution.getExecutionContext().putInt(RESULTS_SIZE_KEY, this.results.size());

        // Total read count
        long readCount = this.jobExecution.getExecutionContext().getLong(TOTAL_READ_COUNT_KEY) + this.results.size();
        this.jobExecution.getExecutionContext().putLong(TOTAL_READ_COUNT_KEY, readCount);

        this.logger.debug("Results size: " + this.results.size());
    }

    /* (non-Javadoc)
     * @see org.springframework.batch.item.database.AbstractPagingItemReader#doJumpToPage(int)
     */
    @Override
    protected void doJumpToPage(int itemIndex)
    {
        // Auto-generated method stub
        // Do nothing.
        // This is for restart the reader and getting back to the page the reader was at.
    }

    abstract protected Collection<? extends T> executePageRead(int page, int pageSize, Map<String, Object> inParameterValues);
}
