package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.BATCH_RUN_ID_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.PROCESSING_FAILURE_STATUS;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemProcessListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.support.ClassifierCompositeItemProcessor;

/**
 *
 * A custom composite / delegate processor for CBSS batch jobs,
 * which will send the data to different processors, depending on
 * different types of data.
 * 
 * Copyright DXC / VA
 * April 12, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public abstract class CBSSDelegateProcessor<I, O> extends ClassifierCompositeItemProcessor<I, O>
                                               implements ItemProcessListener<I, O>, StepExecutionListener
{
    protected Log logger = LogFactory.getLog(getClass());

    protected JobExecution jobExecution;
    protected Long batchRunId;


    @Override
    public void beforeStep(StepExecution stepExecution)
    {
        this.logger.info("Before Step Execution ...");

        // Save the job execution at the beginning of the step.
        this.jobExecution = stepExecution.getJobExecution();

        // Save the batchRunId at the beginning of the step.
        // It is obtained by the batch prior to the job and passed as a job
        // parameter when the job starts.
        this.batchRunId = this.jobExecution.getJobParameters().getLong(BATCH_RUN_ID_KEY);
    }

    @Override
    public ExitStatus afterStep(StepExecution stepExecution)
    {
        this.logger.info("After Step Execution.");

        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            return ExitStatus.FAILED;
        }

        return ExitStatus.COMPLETED;
    }

    @Override
    public void beforeProcess(final I item)
    {
        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            this.logger.error("Before process: System failure detected.");

            if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY))
            {
                this.logger.error("Failure message: " + this.jobExecution.getExecutionContext().get(JOB_FAILURE_MESSAGE_KEY));
            }
        }
    }

    @Override
    public void afterProcess(final I item, final O result)
    {
        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            this.logger.error("After process: System failure detected.");

            if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY))
            {
                this.logger.error("Failure message: " + this.jobExecution.getExecutionContext().get(JOB_FAILURE_MESSAGE_KEY));
            }
        }
    }

    @Override
    public void onProcessError(final I item, Exception e)
    {
        this.logger.error("Exception during processing: ", e);

        StringBuilder error = new StringBuilder();

        error.append("Unable to process item because of ");
        error.append(e.getClass().getSimpleName());
        error.append("\nMessage: ");
        error.append(e.getMessage());

        if ((e.getCause() != null) && (e.getCause().getMessage() != null))
        {
            error.append("\nCause: ");
            error.append(e.getCause().getMessage().trim());
        }

        // Set failure and message.
        stopJob(PROCESSING_FAILURE_STATUS, "Unrecoverable processor error.");
    }

    /**
     * Forcefully stop the job processing because a failure was detected.
     * 
     * @param status
     *            The status for the failure.
     * @param message
     *            The message associated with the status failure.
     */
    protected void stopJob(final String status, final String message)
    {
        // Log message.
        this.logger.error("Processor execution encountered unrecoverable error and forced to stop.");

        // Set failure and message.
        setFailureStatus(status);

        // Set failure message.
        setFailureMessage(message);
    }

    /**
     * Set the failure in the job execution context.
     * 
     * @param status
     *            The failure status.
     */
    protected void setFailureStatus(final String status)
    {
        // Log job failure status.
        this.logger.error("Job failed with status: " + status);

        // Set job failure.
        this.jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
    }

    /**
     * Set the failure message in the job execution context.
     * 
     * @param message
     *            The message to associate with the error status.
     */
    protected void setFailureMessage(final String message)
    {
        // Log job failure message.
        this.logger.error("Job failure message: " + message);

        // Set job failure message.
        this.jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
    }
}
