package gov.va.cpss.job;

import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.JOB_FAILURE_MESSAGE_KEY;
import static gov.va.cpss.job.CbssJobProcessingConstants.WRITE_FAILURE_STATUS;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.ItemWriteListener;
import org.springframework.batch.core.JobExecution;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;

import org.springframework.batch.item.ItemWriter;

/**
 * 
 * CBSS implementation of ItemWriter used to handle writing of raw records.
 * 
 * Copyright HPE / VA
 * February 3, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings("nls")
public abstract class CBSSBaseMultiItemWriter<T> implements ItemWriter<T>, ItemWriteListener<T>, StepExecutionListener
{
    protected Log logger = LogFactory.getLog(getClass());

    /*
     * Flag to indicate that the job has been forcefully stopped and should no
     * longer attempt processing.
     */
    protected boolean isForceStop = false;

    protected JobExecution jobExecution;


    @Override
    public void beforeStep(StepExecution stepExecution)
    {
        this.logger.info("Before Step Execution");

        // Save the job execution at the beginning of the step.
        // The execution context will be used to set key values as data is processed.
        this.jobExecution = stepExecution.getJobExecution();
    }

    @Override
    public ExitStatus afterStep(StepExecution stepExecution)
    {
        this.logger.info("After Step Execution.");

        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            return ExitStatus.FAILED;
        }

        return ExitStatus.COMPLETED;
    }

    @Override
    public void beforeWrite(final List<? extends T> items)
	{
		this.isForceStop = false;
		
		if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
		{
			this.logger.error("Before write: System failure detected.");

            if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY))
            {
                this.logger.error("Failure message: " + this.jobExecution.getExecutionContext().get(JOB_FAILURE_MESSAGE_KEY));
            }

			this.isForceStop = true;
		}
	}

	@Override
    public void afterWrite(final List<? extends T> items)
	{
        if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_KEY))
        {
            this.logger.error("After write: System failure detected.");

            if (this.jobExecution.getExecutionContext().containsKey(JOB_FAILURE_MESSAGE_KEY))
            {
                this.logger.error("Failure message: " + this.jobExecution.getExecutionContext().get(JOB_FAILURE_MESSAGE_KEY));
            }
        }
	}

	@Override
    public void onWriteError(Exception e, final List<? extends T> items)
	{
		this.logger.error("Writer encountered system error and forced stop");

		StringBuilder error = new StringBuilder();

		error.append("Unable to write item because of ");
		error.append(e.getClass().getSimpleName());
		error.append("\nMessage: ");
		error.append(e.getMessage());

		if ((e.getCause() != null) && (e.getCause().getMessage() != null))
		{
			error.append("\nCause: ");
			error.append(e.getCause().getMessage().trim());
		}

		// Set failure and message.
		stopJob(WRITE_FAILURE_STATUS, "Unrecoverable writer error.");
	}


	/**
	 * Forcefully stop the job processing because a failure was detected.
	 * 
	 * @param status
	 *            The status for the failure.
	 * @param message
	 *            The message associated with the status failure.
	 */
	protected void stopJob(final String status, final String message)
	{
        // Log message.
        this.logger.error("Writer execution encountered unrecoverable error and forced to stop.");

		// Set the flag to indicate the job has been forcefully stopped.
		this.isForceStop = true;

		// Set failure and message.
		setFailureStatus(status);

		// Set failure message.
		setFailureMessage(message);
	}

	/**
	 * Set the failure in the job execution context.
	 * 
	 * @param status
	 *            The failure status.
	 */
	protected void setFailureStatus(final String status)
	{
		// Log job failure status.
		this.logger.error("Job failed with status: " + status);

		// Set job failure.
		this.jobExecution.getExecutionContext().putString(JOB_FAILURE_KEY, status);
	}

	/**
	 * Set the failure message in the job execution context.
	 * 
	 * @param message
	 *            The message to associate with the error status.
	 */
	protected void setFailureMessage(final String message)
	{
		// Log job failure message.
		this.logger.error("Job failure message: " + message);

		// Set job failure message.
		this.jobExecution.getExecutionContext().putString(JOB_FAILURE_MESSAGE_KEY, message);
	}
}
