package gov.va.cpss.job;


import static gov.va.cpss.job.CbssJobProcessingConstants.DEFAULT_FILE_DATA_EXTENSION;
import static gov.va.cpss.job.CbssJobProcessingConstants.DEFAULT_FILE_DONE_EXTENSION;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.springframework.batch.core.job.flow.FlowJob;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.cpss.model.BatchRun;
import gov.va.cpss.service.SftpService;

public abstract class AbstractProcessFileListJob extends CbssBaseJob {

	/*
	 * The sftp service.
	 */
	@Autowired
	private SftpService sftpService;

	/**
	 * The file path used to run the job.
	 */
	protected String filename;

	/**
	 * The amount of files that are found on the FTP server.
	 */
	private int fileCount;

	/**
	 * A list of file names found on the server that could not be processed.
	 */
	private List<String> failedFileList;

	/**
	 * The amount of files that were successfully processed.
	 */
	private int successfulFileCount;

	public abstract String getDataDirectory();
	public abstract String getArchiveSubDirectory();
	public abstract String getErrorSubDirectory();

	/**
	 * This should / may not be an abstract method, since it will
	 * break all its subclasses.
	 * 
	 * Using an empty method instead, and any subclass can override
	 * it if it needs.
	 * 
	 * Yiping Yao - 02/23/2017
	 * 
	 * @return
	 */
	//public abstract String getProcessListErrorMessage();
    public String getProcessListErrorMessage()
    {
        // Auto-generated method stub
        return "";
    }

	public String getArchiveDirectory() {
		return getDataDirectory() + "/" + getArchiveSubDirectory();
	}

	public String getErrorDirectory() {
		return getDataDirectory() + "/" + getErrorSubDirectory();
	}

	public String getIndicatorExtension() {
		return DEFAULT_FILE_DONE_EXTENSION;
	}

	public String getDataExtension() {
		return DEFAULT_FILE_DATA_EXTENSION;
	}

	public int getFileCount() {
		return fileCount;
	}

	public List<String> getFailedFileList() {
		return failedFileList;
	}

	public int getSuccessfulFileCount() {
		return successfulFileCount;
	}

	@Override
	protected boolean runJob(BatchRun bR, FlowJob job) {

		boolean success = true;

		List<String> processList = null;
		try {
			processList = getProcessList();
		} catch (Exception e) {
			final String processListErrorMessage = getProcessListErrorMessage();

			// Yiping Yao - 02/23/2017
			// Changed error message - not sure why?
			if (processListErrorMessage != null && !processListErrorMessage.isEmpty())
			{
    			StringBuilder errorBuilder = new StringBuilder(processListErrorMessage);
    			errorBuilder.append("\n").append(e.getMessage()).append("\n");
    			for (Throwable cause = e.getCause(); cause != null ; cause = cause.getCause()) {
    				errorBuilder.append(e.getMessage()).append("\n");
    			}
			
    			jobLogger.error(errorBuilder.toString());
    			super.appendErrorMessage(processListErrorMessage);
			}
			// For previous jobs compatibility.
			else
			{
	            StringBuilder errorBuilder = new StringBuilder("Unable to retrieve list of files from the ftp server:\n");
	            errorBuilder.append(e.getMessage()).append("\n");
	            for (Throwable cause = e.getCause(); cause != null ; cause = cause.getCause()) {
	                errorBuilder.append(e.getMessage()).append("\n");
	            }
	            
	            jobLogger.error(errorBuilder.toString());
	            super.appendErrorMessage(errorBuilder.toString());
			}

			success = false;
		}
		
		// Process the list of files via sftp.
		// Each file will be processed independently.
		if (success) {
			success = processList(bR.getId(), job, processList);
		}
		return success;
	}
	
	protected boolean runJobBypassFtp(BatchRun bR, FlowJob job) {
		List<String> processList = new ArrayList<>();
		processList.add(filename);
		
		return processList(bR.getId(), job, processList);
	}

	/**
	 * Process each file found on the server. Each file is attempted regardless
	 * if there is a failed file.
	 * 
	 * @param batchRunId
	 *            The batchrun primary key id.
	 * @param job
	 *            The job to use to process the data.
	 * @param fileL
	 *            The list of filenames on the server to process.
	 * @return Boolean value indicating success or failed processing.
	 */
	protected boolean processList(final int batchRunId, FlowJob job, List<String> fileL) {

		boolean successful = true;

		// If no file list found then return an error.
		if ((fileL == null) || fileL.isEmpty()) {
			final String error = "No files found on ftp server to process";
			jobLogger.error(error);
			super.appendErrorMessage(error);
			successful = false;
		} else {
			fileCount = fileL.size();
			failedFileList = new ArrayList<String>();
			// Otherwise, perform a batch job for each file found on the server.
			for (String file : fileL) {

				// Build the input file from the indicator file.
				filename = buildDataFilename(file);

				// Only attempt to process if the filename is not null and not
				// empty.
				if ((filename != null) && !filename.isEmpty()) {

					// Process the filename.
					final boolean executionSuccessful = processFile(filename, batchRunId, job);

					// Move the filename to the appropriate archive directory
					// based on the execution processing result.
					final boolean archiveSuccessful = moveFileToArchive(executionSuccessful);

					// Set the combined list processing status if there is any
					// unsuccessful processing of file.
					if (!executionSuccessful || !archiveSuccessful) {
						failedFileList.add(filename);
						successful = false;
					}
				} else {
					jobLogger.error("Attempted to process an invalid filename that was null or empty string");
				}

				// Attempt to remove the indicator file.
				// If there is a problem just log the error but allow processing
				// to continue.
				if (!removeIndicatorFile(file)) {
					jobLogger.error("Problem removing indicator file: " + file);
				}

			}
			successfulFileCount = fileCount - failedFileList.size();
		}

		return successful;
	}

	/**
	 * Get the list of files to process from the ftp server.
	 * 
	 * @return List of filenames on the server.
	 */
	protected List<String> getProcessList() {
		jobLogger.debug("Getting list of files to process: getDataDirectory()=" + getDataDirectory());

		// Use the ftp service to obtain the file list on the server.
		return sftpService.ftpGetFileListWithExtensionInDirectory(getDataDirectory(), getIndicatorExtension(), true);
	}

	/**
	 * Process the specified file.
	 * 
	 * @param file
	 *            The file to process.
	 * @param batchRunId
	 *            The batchrun primary key id.
	 * @param job
	 *            The job to use to process the data.
	 * @return Boolean value indicating success or failed processing.
	 */
	protected abstract boolean processFile(final String file, final int batchRunId, FlowJob job);

	/**
	 * Log error prepending the filename currently being processed.
	 */
	@Override
	protected void logExecuteJobException(final String message) {
		jobLogger.error(filename + ": " + message);
	}

	/**
	 * Append the error message. This allows capture of multiple errors over the
	 * course of multiple file processing.
	 */
	@Override
	protected void appendErrorMessage(final String message) {
		StringBuilder errorBuilder = new StringBuilder();
		errorBuilder.append((new File(filename)).getName());
		errorBuilder.append(": ");
		errorBuilder.append(message);
		super.appendErrorMessage(errorBuilder.toString());
	}

	/**
	 * Move the file to the appropriate archive based on processing status.
	 */
	protected boolean moveFileToArchive(final boolean processingStatus) {

		boolean successful = false;

		// Move file to appropriate directory based on processing status.
		String targetDirectory = getArchiveDirectory();
		if (!processingStatus) {
			targetDirectory = getErrorDirectory();
		}

		if (sftpService.ftpMoveFileFromDirectoryToDirectory(filename, getDataDirectory(), targetDirectory)) {

			// Verify the file was moved properly.
			if (!sftpService.ftpFileExistsInDirectory(filename, getDataDirectory())
					&& sftpService.ftpFileExistsInDirectory(filename, targetDirectory)) {
				successful = true;
			} else {
				appendErrorMessage("Archive of file had unexpected results");
			}

		} else {
			appendErrorMessage("Problem archiving file");
		}

		return successful;
	}

	/**
	 * Build a data filename to process based on the name of the indicator
	 * filename.
	 * 
	 * @param filename
	 *            The indicator filename.
	 * @return The filename of the data file to attempt to locate and process.
	 */
	protected String buildDataFilename(final String filename) {

		try {
			return filename.substring(0, filename.lastIndexOf(getIndicatorExtension())) + getDataExtension();
		} catch (IndexOutOfBoundsException e) {
			jobLogger.error("Unable to get associated data filename from filename: " + filename);
		}

		return null;
	}

	/**
	 * Remove the indicator filename on the sftp server.
	 * 
	 * @param filename
	 *            The filename to remove from the sftp server.
	 * @return Boolean indicating successful or not.
	 */
	protected boolean removeIndicatorFile(final String filename) {
		return sftpService.ftpRemoveFileFromDirectory(filename, getDataDirectory());
	}
}