package gov.va.cpss.jasper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * The Scan Line is the string of characters in the coupon portion of the
 * statement. The Scan Line is used by Lock Box for routing payments. The Scan
 * Line logic is provided to us from AITC in Cobol and translated to Java.
 * 
 * @author DNS  
 */
public class ScanLine {

	// The size of the scan line happens to be the same as the answer to life.
	private static final int CHECK_DIGIT_ARRAY_SIZE = 42;

	// Reserved size of facility.
	private static final int FACILITY_NUMBER_RESERVED_LENGTH = 5;

	// Reserved size of account number.
	private static final int ACCOUNT_NUMBER_RESERVED_LENGTH = 24;

	// Reserved miscellaneous string is always four zeros.
	private static final String MISCELLANEOUS_NUMBER_RESERVED_VALUE = "0000";

	// Reserved size of amount due.
	private static final int AMOUNT_DUE_RESERVED_LENGTH = 9;

	// The expected size of the temporary weight number.
	private static final int TEMPORARY_WEIGHT_NUMBER_LENGTH = 2;

	// The expected size of the check digit total.
	private static final int CHECK_DIGIT_TOTAL_LENGTH = 4;

	// Initialized to 0 as per Cobol 8100 algorithm.
	private static final int CHECK_DIGIT_SUM_INITIAL_VALUE = 0;

	// Initialized to 2 as per Cobol 8100 algorithm.
	private static final int WEIGHT_NUMBER_INITIAL_VALUE = 2;

	// The temporary scan line array that has been sanitized to remove invalid
	// characters.
	private char[] scanLineA = new char[CHECK_DIGIT_ARRAY_SIZE];

	// The weight number used during calculation.
	private int weightNumber = WEIGHT_NUMBER_INITIAL_VALUE;

	// The check digit sum used during calculation.
	private int checkDigitSum = CHECK_DIGIT_SUM_INITIAL_VALUE;

	// An array of integers representing the associated integer values mapped
	// for the scan line array.
	private int[] checkDigitDataPos = new int[CHECK_DIGIT_ARRAY_SIZE];

	// An array of weights that were calculated from the checkDigitDataPos.
	private int[] weightUpdatedAns = new int[CHECK_DIGIT_ARRAY_SIZE];

	// The resulting scan check digit that results from the algorithm
	// calculation.
	private int scanCheckDigit = 0;

	// The resulting scan line. It is the original data with the trailing scan
	// check digit.
	private String calculatedScanLine = null;

	// The infamous lookup map.
	private static Map<Character, Integer> conversionTableM;
	{
		Map<Character, Integer> conversionTableInitM = new HashMap<Character, Integer>();
		// Alphabet
		conversionTableInitM.put('A', 0);
		conversionTableInitM.put('B', 1);
		conversionTableInitM.put('C', 2);
		conversionTableInitM.put('D', 3);
		conversionTableInitM.put('E', 4);
		conversionTableInitM.put('F', 5);
		conversionTableInitM.put('G', 6);
		conversionTableInitM.put('H', 7);
		conversionTableInitM.put('I', 8);
		conversionTableInitM.put('J', 9);
		conversionTableInitM.put('K', 0);
		conversionTableInitM.put('L', 1);
		conversionTableInitM.put('M', 2);
		conversionTableInitM.put('N', 3);
		conversionTableInitM.put('O', 4);
		conversionTableInitM.put('P', 5);
		conversionTableInitM.put('Q', 6);
		conversionTableInitM.put('R', 7);
		conversionTableInitM.put('S', 8);
		conversionTableInitM.put('T', 9);
		conversionTableInitM.put('U', 0);
		conversionTableInitM.put('V', 1);
		conversionTableInitM.put('W', 2);
		conversionTableInitM.put('X', 3);
		conversionTableInitM.put('Y', 4);
		conversionTableInitM.put('Z', 5);
		// Symbols
		conversionTableInitM.put('`', 0);
		conversionTableInitM.put('!', 1);
		conversionTableInitM.put('@', 2);
		conversionTableInitM.put('#', 3);
		conversionTableInitM.put('$', 4);
		conversionTableInitM.put('%', 5);
		conversionTableInitM.put('^', 6);
		conversionTableInitM.put('&', 7);
		conversionTableInitM.put('*', 8);
		conversionTableInitM.put('(', 9);
		conversionTableInitM.put(')', 0);
		conversionTableInitM.put('-', 1);
		conversionTableInitM.put('+', 2);
		conversionTableInitM.put('=', 3);
		conversionTableInitM.put(';', 4);
		conversionTableInitM.put(':', 5);
		conversionTableInitM.put('\'', 6);
		conversionTableInitM.put('"', 7);
		conversionTableInitM.put(',', 8);
		conversionTableInitM.put('.', 9);
		conversionTableInitM.put('/', 0);
		conversionTableInitM.put('\\', 1);
		conversionTableInitM.put('<', 2);
		conversionTableInitM.put('>', 3);
		conversionTableInitM.put('?', 4);
		conversionTableInitM.put(']', 5);
		conversionTableInitM.put('_', 6);
		conversionTableInitM.put('{', 7);
		conversionTableInitM.put('}', 8);
		conversionTableInitM.put('[', 9);
		conversionTableInitM.put('\u00A2', 0); // cent symbol
		conversionTableInitM.put('~', 5);
		// Numbers
		// conversionTableInitM.put('0', 0);
		// conversionTableInitM.put('1', 3);
		// conversionTableInitM.put('2', 6);
		// conversionTableInitM.put('3', 9);
		// conversionTableInitM.put('4', 2);
		// conversionTableInitM.put('5', 6);
		// conversionTableInitM.put('6', 9);
		// conversionTableInitM.put('7', 2);
		// conversionTableInitM.put('8', 5);
		// conversionTableInitM.put('9', 8);

		conversionTableInitM.put('0', 0);
		conversionTableInitM.put('1', 1);
		conversionTableInitM.put('2', 2);
		conversionTableInitM.put('3', 3);
		conversionTableInitM.put('4', 4);
		conversionTableInitM.put('5', 5);
		conversionTableInitM.put('6', 6);
		conversionTableInitM.put('7', 7);
		conversionTableInitM.put('8', 8);
		conversionTableInitM.put('9', 9);

		conversionTableM = Collections.unmodifiableMap(conversionTableInitM);
	}

	/**
	 * Construct a scan line from the site number, account number, and amount
	 * due. The constructor will run the algorithm at construction to create the
	 * resulting scan line. Creation of new object should be wrapped in a try
	 * catch block in case of invalid inputs.
	 * 
	 * @param siteNumber
	 *            The facility or site number.
	 * @param accountNumber
	 *            The account number.
	 * @param amountDue
	 *            The amount due.
	 */
	public ScanLine(final String siteNumber, final String accountNumber, final String amountDue) {

		StringBuilder builder = new StringBuilder();

		// Build a scan line string based on the input values.
		builder.append(rightPad(siteNumber.trim(), FACILITY_NUMBER_RESERVED_LENGTH, '*'));
		builder.append(rightPad(accountNumber.trim(), ACCOUNT_NUMBER_RESERVED_LENGTH, '*'));
		builder.append(MISCELLANEOUS_NUMBER_RESERVED_VALUE);
		final String amountDueS = amountDue.trim().replace(".", "");
		builder.append(leftPad(amountDueS, AMOUNT_DUE_RESERVED_LENGTH, '0'));

		final String scanLine = builder.toString();

		if (scanLine.length() != CHECK_DIGIT_ARRAY_SIZE) {
			throw new IndexOutOfBoundsException("Scan Line (" + scanLine + ") has invalid length (" + scanLine.length()
					+ ") but expected (" + CHECK_DIGIT_ARRAY_SIZE + ")");
		}

		// Convert the scan line to an array while also cleaning the scan line
		// of invalid characters.
		convertScanLineToArray(scanLine);

		// Build the check digit data array from the scan line.
		populateCheckDigitData();

		// Compute the scan check digit;
		computeScanCheckDigit();

		// Build the completed scan line.
		buildScanLine();
	}

	/**
	 * Get the calculated scan line.
	 * 
	 * @return The calculated scan line.
	 */
	public String getCalculatedScanLine() {
		return calculatedScanLine;
	}

	/**
	 * Build a scan line array from the input scan line while also sanitizing
	 * the scan line characters of invalid space or double quote characters.
	 * 
	 * @param scanLine
	 *            The original scan line.
	 */
	private void convertScanLineToArray(final String scanLine) {
		for (int i = 0; i < CHECK_DIGIT_ARRAY_SIZE; ++i) {
			char c = scanLine.charAt(i);
			switch (c) {
			case ' ':
				c = '*';
				break;
			case '"':
				c = '/';
				break;
			default:
				break;
			}
			scanLineA[i] = c;
		}
	}

	/**
	 * Build a check digit array associated with the scan line array.
	 */
	private void populateCheckDigitData() {

		for (int i = 0; i < CHECK_DIGIT_ARRAY_SIZE; ++i) {
			checkDigitDataPos[i] = lookupNumberMapping(scanLineA[i]);
		}
	}

	/**
	 * Compute the scan check digit. This is cobol algorithm 8100.
	 */
	// 8100-COMPUTE-SCAN-CHECK-DIGIT.
	// MOVE WS-NONE TO CHECK-DIGIT-SUM.
	// MOVE 2 TO WEIGHT-NUMBER.
	// PERFORM 8400-CHECK-DIGIT-FORMULA-1 VARYING SCAN-SUB FROM
	// ONE BY ONE UNTIL SCAN-SUB > FORTY-TWO.
	// PERFORM 8500-CHECK-DIGIT-FORMULA-2 VARYING SCAN-SUB FROM
	// ONE BY ONE UNTIL SCAN-SUB > FORTY-TWO.
	// PERFORM 8700-CHECK-DIGIT-FORMULA-3.
	private void computeScanCheckDigit() {

		checkDigitSum = CHECK_DIGIT_SUM_INITIAL_VALUE;
		weightNumber = WEIGHT_NUMBER_INITIAL_VALUE;

		// Perform 8400
		for (int i = 0; i < CHECK_DIGIT_ARRAY_SIZE; ++i) {
			checkDigitFormula1(i);
		}

		// Perform 8500
		for (int j = 0; j < CHECK_DIGIT_ARRAY_SIZE; ++j) {
			checkDigitFormula2(j);
		}

		// Perform 8700
		checkDigitFormula3();
	}

	/**
	 * Compute the scan check digit formula 1. This is cobol algorithm 8400.
	 */
	// 8400-CHECK-DIGIT-FORMULA-1.
	// COMPUTE TEMP-WEIGHT-NUM = CHECK-DIGIT-DATA-POS (SCAN-SUB)
	// * WEIGHT-NUMBER.
	// ADD WEIGHT-NUM-1 TO WEIGHT-NUM-2.
	// COMPUTE WEIGHT-NUMBER = WEIGHT-NUMBER - 1.
	// MOVE WEIGHT-NUM-2 TO WEIGHT-UPDATED-ANS (SCAN-SUB).
	// IF WEIGHT-NUMBER < 1
	// MOVE 2 TO WEIGHT-NUMBER.
	private void checkDigitFormula1(final int scanSub) {

		final int tempWeightNum = checkDigitDataPos[scanSub] * weightNumber;

		final List<Integer> tempWeightNumL = convertValueToList(tempWeightNum, TEMPORARY_WEIGHT_NUMBER_LENGTH);

		final int addedWeightNum = tempWeightNumL.get(0) + tempWeightNumL.get(1);

		weightNumber = weightNumber - 1;

		weightUpdatedAns[scanSub] = addedWeightNum;

		if (weightNumber < 1) {
			weightNumber = WEIGHT_NUMBER_INITIAL_VALUE;
		}
	}

	/**
	 * Compute the scan check digit formula 2. This is cobol algorithm 8500.
	 */
	// 8500-CHECK-DIGIT-FORMULA-2.
	// ADD WEIGHT-UPDATED-ANS (SCAN-SUB) TO CHECK-DIGIT-SUM.
	private void checkDigitFormula2(final int scanSub) {
		checkDigitSum += weightUpdatedAns[scanSub];
	}

	/**
	 * Compute the scan check digit formula 3. This is cobol algorithm 8700.
	 */
	// 8700-CHECK-DIGIT-FORMULA-3.
	// MOVE CHECK-DIGIT-SUM TO CHECK-DIGIT-TOTAL.
	// IF CHECK-DIGIT-UNITS-POS NOT = ZERO
	// SUBTRACT CHECK-DIGIT-UNITS-POS FROM 10
	// GIVING SCAN-CHECK-DIGIT
	// ELSE
	// MOVE CHECK-DIGIT-UNITS-POS TO SCAN-CHECK-DIGIT.
	private void checkDigitFormula3() {

		final List<Integer> checkDigitTotalL = convertValueToList(checkDigitSum, CHECK_DIGIT_TOTAL_LENGTH);

		final int checkDigitUnitsPos = checkDigitTotalL.get(3);

		if (checkDigitUnitsPos > 0) {
			scanCheckDigit = 10 - checkDigitUnitsPos;
		}
	}

	/**
	 * Build the resulting calculated scan line by appending the scan check
	 * digit to the scan line array.
	 */
	private void buildScanLine() {
		calculatedScanLine = String.valueOf(scanLineA) + String.valueOf(scanCheckDigit);
	}

	/**
	 * Look up the associated number for the input character. If the character
	 * is not recognized in the map the associated value is zero.
	 * 
	 * @param c
	 *            The character to look up in the map.
	 * @return The associated number value for the character.
	 */
	private int lookupNumberMapping(final char c) {
		final Integer value = conversionTableM.get(c);
		if (value == null) {
			return 0;
		}
		return value;
	}

	/**
	 * Return the specified string that is left padded by the specified pad
	 * character that is the specified length.
	 * 
	 * @param s
	 *            The string to pad.
	 * @param nb
	 *            The total length of the desired padded string.
	 * @param pad
	 *            The character to use as a pad.
	 * @return The padded string.
	 */
	private static String leftPad(String s, int nb, char pad) {
		return Optional.of(nb - s.length()).filter(i -> i > 0)
				.map(i -> String.format("%" + i + "s", "").replace(" ", pad + "") + s).orElse(s);
	}

	/**
	 * Return the specified string that is right padded by the specified pad
	 * character that is the specified length.
	 * 
	 * @param s
	 *            The string to pad.
	 * @param nb
	 *            The total length of the desired padded string.
	 * @param pad
	 *            The character to use as a pad.
	 * @return The padded string.
	 */
	private static String rightPad(String s, int nb, char pad) {
		return Optional.of(nb - s.length()).filter(i -> i > 0)
				.map(i -> s + String.format("%" + i + "s", "").replace(" ", pad + "")).orElse(s);
	}

	/**
	 * Convert an integer to a list of integers. Easiest to explain by examples:
	 * 1. Input value of 1234 and pad size of 4 results in list of Integer {1,
	 * 2, 3, 4} 2. Input value of 12 and pad size of 4 results in list of
	 * Integer {0, 0, 1, 2}
	 * 
	 * @param value
	 *            The integer to convert.
	 * @param padSize
	 *            The pad size that specifies the size of the returned list.
	 * @return List of Integers that represent the passed in value.
	 * @throws NumberFormatException
	 */
	public static List<Integer> convertValueToList(int value, final int padSize) throws NumberFormatException {

		// How many digits are in the value?
		int length = 1;
		if (value > 0) {
			length = (int) (Math.log10(value) + 1);
		}

		// If too big throw exception.
		if (length > padSize) {
			throw new NumberFormatException("Value larger than expected: " + value);
		}

		List<Integer> valueL = new ArrayList<>();

		do {
			valueL.add(value % 10);
			value /= 10;
		} while (value > 0);

		// If bigger than expected size, throw exception.
		if (valueL.size() > padSize) {
			throw new NumberFormatException("Failed to convert to List: " + value);
		}

		// Pad if needed to expected length.
		while (valueL.size() < padSize) {
			valueL.add(0);
		}

		// Flip the order.
		Collections.reverse(valueL);

		return valueL;
	}
}
