package gov.va.cpss.jasper;

import java.awt.Image;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;

import org.apache.log4j.Logger;

import com.ibm.icu.util.Calendar;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSSiteStmtDAO;
import gov.va.cpss.dao.CBSSiteTransDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.job.sendcbs.SendCBSRuntimeStateImpl;
import gov.va.cpss.job.sendcbs.SendCBSWritePSRecordSource;
import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSSiteTrans;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.fps.PSDetails;
import gov.va.cpss.model.fps.PSPatient;
import gov.va.cpss.model.fps.PSRecord;
import gov.va.cpss.model.fps.PSSite;
import gov.va.cpss.model.fps.PSSiteStmt;
import gov.va.cpss.model.updatestation.Address;
import gov.va.cpss.model.updatestation.StationInfo;
import gov.va.cpss.service.StationInfoUtil;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;

/**
 * Service class to create a PDF for a CBSStmt record.
 * 
 * @author DNS   
 */
@SuppressWarnings({"nls", "static-method"})
public class PdfPatientStatementFactoryService {

	private static final Logger logger = Logger.getLogger(PdfPatientStatementFactoryService.class.getCanonicalName());

	private ClassLoader classLoader = PdfPatientStatementFactoryService.class.getClassLoader();

	// AITC template specifies that the first page should not have more than
	// this many lines.
	private static final int MAX_LENGTH_FIRST_PAGE = 22;

	// And the subsequent pages should not have more than this many lines.
	private static final int MAX_LENGTH_EXTENSION_PAGE = 39;

	//
	// The account number is split on the report.
	//
	// First 3 digits / characters for Facility / Station Number
	private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_1 = 3;
	private static final int ACCOUNT_NUMBER_SPLIT_POSITION_1 = 0;

	// Fix for Defect 616442
	// - Reformat Account Number
	// Yiping Yao - 01/30/2018

	// DFN 1st 4 digits
	private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_2 = 4;
	private static final int ACCOUNT_NUMBER_SPLIT_POSITION_2 = ACCOUNT_NUMBER_SPLIT_POSITION_1 +
															   ACCOUNT_NUMBER_SPLIT_LENGTH_1;
	//private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_2 = 3;

	// DFN 2nd 4 digits
	private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_3 = 4;
	private static final int ACCOUNT_NUMBER_SPLIT_POSITION_3 = ACCOUNT_NUMBER_SPLIT_POSITION_2 +
															   ACCOUNT_NUMBER_SPLIT_LENGTH_2;
	//private static final int ACCOUNT_NUMBER_SPLIT_POSITION_3 = 6;
	//private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_3 = 3;

	// DFN last 5 digits
	private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_4 = 5;
	private static final int ACCOUNT_NUMBER_SPLIT_POSITION_4 = ACCOUNT_NUMBER_SPLIT_POSITION_3 +
															   ACCOUNT_NUMBER_SPLIT_LENGTH_3;
	//private static final int ACCOUNT_NUMBER_SPLIT_POSITION_4 = 9;
	//private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_4 = 4;

	// Last Name 1st 5 characters
	private static final int ACCOUNT_NUMBER_SPLIT_LENGTH_5 = 5;
	private static final int ACCOUNT_NUMBER_SPLIT_POSITION_5 = ACCOUNT_NUMBER_SPLIT_POSITION_4 +
															   ACCOUNT_NUMBER_SPLIT_LENGTH_4;
	// The length of this is optional, depending on the length of last name.
	//private static final int ACCOUNT_NUMBER_SPLIT_POSITION_5 = 13;

	// String constant characters
	private static final String STRING_SPACE = " ";
	private static final String STRING_DASH = "-";
	private static final String EMPTY_STRING = "";


	// Dates on report are formatted MM/dd/yyyy
	private static final String REPORT_DATE_FORMAT = "MM/dd/yyyy";

	// Number of days to add to statement date to get the full payment date.
	private static final int FULL_PAYMENT_DAY_NUMBER_DELTA = 25;

	// Credit values have trailing CR
	private static final String CREDIT_SUFFIX = "CR";

	// HTML tag / entity
	private static final String HTML_LINE_BREAK = "<br/>";
	private static final String HTML_NON_BREAKING_SPACE = "&nbsp;";

	// Property / Tag Names and Values
	private static final String VA_LOGO = "vaLogo";
	private static final String VA_LOGO_RESOURCE = "7_VA_1COLOR_HORIZONT_12C9F6.jpg";

	private static final String CHECKBOX ="checkbox";
	private static final String CHECKBOX_RESOURCE = "checkbox.png";

	private static final String STATEMENT_DATE = "StatementDate";
	private static final String PAYMENT_DATE = "PaymentDate";
	private static final String PAYMENT_DUE_DATE = "PaymentDueDate";
	private static final String PAYMENT_DUE_MESSAGE = "PaymentDueMessage";
	private static final String PAYMENT_DUE_MESSAGE_TEXT = "TO AVOID LATE" + HTML_LINE_BREAK +
														   "CHARGES PAY BALANCE" + HTML_LINE_BREAK + "BY ";
	private static final String NO_PAYMENT_DUE_MESSAGE_TEXT = "NO PAYMENT DUE";
	private static final String BALANCE = "Balance";
	private static final String AMOUNT_DUE = "AmountDue";
	private static final String PATIENT_NAME = "PatientName";
	private static final String PATIENT_ADDRESS = "PatientAddress";
	private static final String PREVIOUS_BALANCE = "PreviousBalance";
	private static final String NEW_CHARGES = "NewCharges";
	private static final String PAYMENTS_RECEIVED = "PaymentsReceived";
	private static final String ACCOUNT_NUMBER_RAW = "AccountNumberRaw";
	private static final String ACCOUNT_NUMBER = "AccountNumber";
	private static final String LOCAL_MESSAGE = "LocalMessage";
	private static final String LOCAL_MESSAGE_TEXT = "LOCAL VA'S MESSAGE";
	private static final String LOCAL_MESSAGE_MULTISITE_TEXT = HTML_LINE_BREAK + HTML_NON_BREAKING_SPACE + HTML_NON_BREAKING_SPACE +
															   "* CONSOLIDATED STATEMENT GENERATED" +
															   HTML_LINE_BREAK + HTML_NON_BREAKING_SPACE + HTML_NON_BREAKING_SPACE +
															   "* REVIEW TEAR-OFF COUPON FOR AMOUNT DUE";
	private static final String MAILING_LABEL = "MailingLabel";
	private static final String SITE_LOCATION_NAME = "SiteLocationName";
	private static final String SITE_ADDRESS = "SiteAddress";
	private static final String SITE_CITY_STATE_ZIP = "SiteCityStateZip";
	private static final String QUESTIONS_PHONE = "QuestionsPhone";
	private static final String PAY_BY_PHONE = "PayByPhone";
	private static final String PAY_BY_PHONE_TEXT = "1-888-827-4817";
	private static final String PAY_HOURS = "PayHours";
	private static final String PAY_HOURS_TEXT = "Mon - Fri 8am - 8pm EST";
	private static final String MAIL_TO_ADDRESS = "MailToAddress";
	private static final String MAIL_TO_ADDRESS_TEXT = "DEPARTMENT OF VETERANS AFFAIRS" + HTML_LINE_BREAK +
													   "PO BOX 530269" + HTML_LINE_BREAK +
													   "ATLANTA GA 30353-0269";


	// Properties
	private CBSStmtDAO cbsStmtDAO;

	private CBSSiteStmtDAO cbsSiteStmtDAO;

	private CBSSiteTransDAO cbsSiteTransDAO;

//	private StationInfoUtil stationInfoService;

	private JasperReport mainReport;
	private JasperReport detailSubreport;
	private JasperReport siteTitleSubreport;
	private JasperReport siteTotalSubreport;
	private JasperReport siteDelimiterSubreport;
	private JasperReport siteTrailingSpaceSubreport;
	private JasperReport siteFillerSpaceSubreport;
	private JasperReport detailEntrySubreport;
	private JasperReport siteLateStmtSubreport;

	private Image vaLogo;
	private Image checkbox;

	public CBSStmtDAO getCbsStmtDAO() {
		return this.cbsStmtDAO;
	}

	public void setCbsStmtDAO(CBSStmtDAO inCbsStmtDAO) {
		this.cbsStmtDAO = inCbsStmtDAO;
	}

	public CBSSiteStmtDAO getCbsSiteStmtDAO() {
		return this.cbsSiteStmtDAO;
	}

	public void setCbsSiteStmtDAO(CBSSiteStmtDAO inCbsSiteStmtDAO) {
		this.cbsSiteStmtDAO = inCbsSiteStmtDAO;
	}

	public CBSSiteTransDAO getCbsSiteTransDAO() {
		return this.cbsSiteTransDAO;
	}

	public void setCbsSiteTransDAO(CBSSiteTransDAO inCbsSiteTransDAO) {
		this.cbsSiteTransDAO = inCbsSiteTransDAO;
	}

//	public StationInfoService getStationInfoService() {
//		return stationInfoService;
//	}
//
//	public void setStationInfoService(StationInfoService stationInfoService) {
//		this.stationInfoService = stationInfoService;
//	}
	
	public void initJasperReport(){

		String mainReportJasper = "ConsolidatedPatientStatement.jrxml";
		String detailSubreportJasper = "CpsSiteSubreport.jrxml";
		String siteTitleSubreportJasper = "CpsSiteTitleSubreport.jrxml";
		String siteTotalSubreportJasper = "CpsSiteTotalSubreport.jrxml";
		String siteDelimiterSubreportJasper = "CpsSiteDelimiterSubreport.jrxml";
		String siteTrailingSpaceSubreportJasper = "CpsSiteTrailingSpaceSubreport.jrxml";
		String siteFillerSpaceSubreportJasper = "CpsSiteFillerSpaceSubreport.jrxml";
		String detailEntrySubreportJasper = "CpsSiteEntrySubreport.jrxml";
		String siteLateStmtJasper = "CpsSiteLateStmtSubreport.jrxml";

		try
		(
			InputStream mainReportIS = this.classLoader.getResourceAsStream(mainReportJasper);
			InputStream detailSubreportIS = this.classLoader.getResourceAsStream(detailSubreportJasper);
			InputStream siteTitleSubreportIS = this.classLoader.getResourceAsStream(siteTitleSubreportJasper);
			InputStream siteTotalSubreportIS = this.classLoader.getResourceAsStream(siteTotalSubreportJasper);
			InputStream siteDelimiterSubreportIS = this.classLoader.getResourceAsStream(siteDelimiterSubreportJasper);
			InputStream siteTrailingSpaceSubreportIS = this.classLoader.getResourceAsStream(siteTrailingSpaceSubreportJasper);
			InputStream siteFillerSpaceSubreportIS = this.classLoader.getResourceAsStream(siteFillerSpaceSubreportJasper);
			InputStream detailEntrySubreportIS = this.classLoader.getResourceAsStream(detailEntrySubreportJasper);
			InputStream siteLateStmtIS = this.classLoader.getResourceAsStream(siteLateStmtJasper);

			InputStream vaLogoIs = this.classLoader.getResourceAsStream(VA_LOGO_RESOURCE);
			InputStream checkboxIs = this.classLoader.getResourceAsStream(CHECKBOX_RESOURCE);
		)
		{
			logger.info("Compiling Jasper Reports");

			this.vaLogo = ImageIO.read(vaLogoIs);
			this.checkbox = ImageIO.read(checkboxIs);
			
			// Compile the main report file to an object to populate
			// dynamically.
			this.mainReport = JasperCompileManager.compileReport(mainReportIS);
			this.detailSubreport = JasperCompileManager.compileReport(detailSubreportIS);
			this.siteTitleSubreport = JasperCompileManager.compileReport(siteTitleSubreportIS);
			this.siteTotalSubreport = JasperCompileManager.compileReport(siteTotalSubreportIS);
			this.siteDelimiterSubreport = JasperCompileManager.compileReport(siteDelimiterSubreportIS);
			this.siteTrailingSpaceSubreport = JasperCompileManager.compileReport(siteTrailingSpaceSubreportIS);
			this.siteFillerSpaceSubreport = JasperCompileManager.compileReport(siteFillerSpaceSubreportIS);
			this.detailEntrySubreport = JasperCompileManager.compileReport(detailEntrySubreportIS);
			this.siteLateStmtSubreport = JasperCompileManager.compileReport(siteLateStmtIS);

		} catch (NumberFormatException e) {
			// Scan Line may throw this.
			logger.error("Number Format Exception: " + e.getMessage());

		} catch (JRException e) {
			// Jasper may throw this.
			logger.error("JR Exception: " + e.getMessage());

		} catch (IOException e) {
			logger.error(e.getMessage());
		}
	}

	/**
	 * Generate a PDF statement for the specified CBSStmt ID
	 * 
	 * @param cbsStmtId
	 *            The CBSStmt ID.
	 * @return pdf
	 * 			The generated pdf as a byte array
	 */
	public byte[] generateStatementPdfStream(final long cbsStmtId) {

		JasperPrint jasperPrint = generateStatementPdfPrint(cbsStmtId);
		byte [] pdf=null;
		if (jasperPrint != null) {
			// Export the dynamically filled report to disk.
			try {
				// Export the dynamically filled report to stream.
				pdf = JasperExportManager.exportReportToPdf(jasperPrint);
			} catch (JRException e) {
				logger.error(e.getMessage());
			}
		} else {
			logger.error("jasperPrint is null");
		}
		return pdf;
	}

	/**
	 * Generate a PDF statement for the specified CBSStmt ID output on the
	 * provided stream.
	 * 
	 * @param cbsStmtId
	 *            The CBSStmt ID.
	 * @param outputFile
	 *            The filename for which to output the PDF.
	 */
	public void generateStatementPdfFile(final long cbsStmtId, final String outputFile) {

		JasperPrint jasperPrint = generateStatementPdfPrint(cbsStmtId);

		if (jasperPrint != null) {
			logger.info("AFTER Generation: " + outputFile);

			// Export the dynamically filled report to disk.
			try {

				JasperExportManager.exportReportToPdfFile(jasperPrint, outputFile);
			} catch (JRException e) {
				logger.error(e.getMessage());
			}
		} else {
			logger.error("Cbs Stmt ID not found: " + cbsStmtId);
		}
	}

	/**
	 * Generate a Jasper Print object for the specified CBSStmt ID.
	 * 
	 * @param cbsStmtId
	 *            The CBSStmt ID.
	 * @return The Jasper Print object.
	 */
	public JasperPrint generateStatementPdfPrint(final long cbsStmtId) {

		try {

			// Convert from CBSStmt
			final SendCBSWritePSRecordSource psRecordSource = buildRecordSource(cbsStmtId);
			final PdfConsolidatedStatementDataBean dataBean = buildStatementDataBean(psRecordSource, cbsStmtId);
			
			// TODO, if above are null or empty then return null!?!?
			
			// Fix for Defect 630398
			// - Remove facility total for single site (hence the delimiter lines)
			// Yiping Yao - 01/19/2018
			boolean isMultiSite = psRecordSource.isMultisiteflag();

			// Get references.
			Map<String, Object> mainReportValueParametersM = dataBean.getMainReportValueParametersM();
			List<PdfSiteStatementCollectionBean> siteStatementCollectionBeanL = dataBean
					.getPdfSiteStatementCollectionBeanL();

			// Create a data source to use as inputs to populate the site
			// statement sub report portion of the PDF.
			JRBeanCollectionDataSource siteStatementCollectionDataSource = new JRBeanCollectionDataSource(
					siteStatementCollectionBeanL);
						
			// Create a parameter map to use as inputs to populate the global
			// portion of the PDF.

			// Add the subreports as parameters.
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteSubreportParameter",
											this.detailSubreport);
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteTitleSubreportParameter",
											this.siteTitleSubreport);
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteTotalSubreportParameter",
											this.siteTotalSubreport);

			// Fix for Defect 630398
			// - Remove facility total for single site (hence the delimiter lines)
			// Yiping Yao - 01/19/2018
			if (isMultiSite)
			{	
				mainReportValueParametersM.put("ConsolidatedPatientStatementSiteDelimiterSubreportParameter",
												this.siteDelimiterSubreport);
			}

			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteTrailingSpaceSubreportParameter", 
											this.siteTrailingSpaceSubreport);
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteFillerSpaceSubreportParameter",
											this.siteFillerSpaceSubreport);
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteEntrySubreportParameter",
											this.detailEntrySubreport);
			mainReportValueParametersM.put("ConsolidatedPatientStatementSiteLateStmtSubreportParameter",
											this.siteLateStmtSubreport);

			mainReportValueParametersM.put(VA_LOGO, this.vaLogo);
			mainReportValueParametersM.put(CHECKBOX, this.checkbox);
			
			// Dynamically fill the report with the parameter map and sub report
			// collection data source.
			JasperPrint jasperPrint = JasperFillManager.fillReport(this.mainReport, mainReportValueParametersM,
					siteStatementCollectionDataSource);

			return jasperPrint;

		} catch (NumberFormatException e) {

			// Scan Line may throw this.
			logger.error(e.getMessage());

		} catch (JRException e) {

			// Jasper may throw this.
			logger.error(e.getMessage());

		} catch (IndexOutOfBoundsException e) {

			// Scan Line may throw this.
			logger.error(e.getMessage());
		}

		return null;
	}

	/**
	 * Build a SendCBSWritePSRecordSource object for the specified CBSStmt. The
	 * returned object is used to populate a Jasper Data Source.
	 * 
	 * @param cbsStmtId
	 *            The CBSStmt ID.
	 * @return The SendCBSWritePSRecordSource.
	 */
	private SendCBSWritePSRecordSource buildRecordSource(final long cbsStmtId) {

		final SendCBSWritePSRecordSource psRecordSource = new SendCBSWritePSRecordSource();

		final SendCBSRuntimeStateImpl runtimeState = new SendCBSRuntimeStateImpl();
        psRecordSource.setSendCBSRuntimeState(runtimeState);

        final CBSStmt cbsStmt = this.cbsStmtDAO.get(cbsStmtId);

        final List<CBSSiteStmt> siteStmtL = this.cbsSiteStmtDAO.getAllByCBSStmtID(cbsStmtId, true);

        final List<CBSSiteTrans> siteTransL = this.cbsSiteTransDAO.getAllByCBSSiteStmtID(cbsStmtId);
        if(siteStmtL.size() > 1) psRecordSource.setMultisiteflag(true);

        for(CBSSiteStmt siteStmt : siteStmtL) {
     	   siteStmt.setSiteTransL(new ArrayList<CBSSiteTrans>());
        }

        for(CBSSiteTrans trans : siteTransL){
     	   for(CBSSiteStmt siteStmt : siteStmtL) {
     		   if(trans.getSiteStmtId() == siteStmt.getId()){
     			   siteStmt.getSiteTransL().add(trans);
     		   }
     	   }
        }

        cbsStmt.setSiteStmtL(siteStmtL);

        final List<PSSite> psSitesList = runtimeState.getPSSitesList();
        psSitesList.add(createPSSite(PSRecord.DataType.PH, 1, 1, siteStmtL.get(0), 1, cbsStmt.getNewBalance().getDouble()));

        final List<CBSStmt> cbsStmtL2 = new ArrayList<>(1);
        cbsStmtL2.add(cbsStmt);

        final Map<String, List<CBSStmt>> cbsListsBySite = runtimeState.getCBSListsBySite();
        cbsListsBySite.put(cbsStmt.getPrimarySiteStmt().getStationNum(), cbsStmtL2);

        psRecordSource.setPsSitesList(psSitesList);
        psRecordSource.setCbsListsBySite(cbsListsBySite);

        return psRecordSource;
	}

	/**
	 * Build a PSSite object from data retrieved from the database.
	 * 
	 * @param statementType
	 * @param psSeqNum
	 * @param psTotSeqNum
	 * @param primarySite
	 * @param psTotStatement
	 * @param psStatementVal
	 * @return The PSSite object.
	 */
	private PSSite createPSSite(final PSRecord.DataType statementType, final int psSeqNum, final int psTotSeqNum,
			final CBSSiteStmt primarySite, final int psTotStatement, final double psStatementVal) {
		PSSite psSite = new PSSite();
		psSite.setType(PSRecord.DataType.PS);
		psSite.setStatementType(statementType);

		psSite.setSeqNum(psSeqNum);
		psSite.setTotSeqNum(psTotSeqNum);
		psSite.setFacilityNum(primarySite.getStationNum());
		psSite.setFacilityPhoneNum(primarySite.getStationPhoneNum());
		psSite.setStatementDate(primarySite.getStatementDate());
		psSite.setProcessDate(primarySite.getProcessDate());
		psSite.setTotStatement(psTotStatement);
		psSite.setStatementVal(new Money(psStatementVal, 11));

		return psSite;
	}

	/**
	 * Create a PdfConsolidatedStatementDataBean of parameters to populate the
	 * PDF report from a SendCBSWritePSRecordSource object.
	 * 
	 * @param psRecordSource
	 *            The SendCBSWritePSRecordSource object containing the requested
	 *            CBSStmt data.
	 * @param cbsStmtId 
	 * @return The PdfConsolidatedStatementDataBean for the Jasper Report.
	 */
	private PdfConsolidatedStatementDataBean buildStatementDataBean(final SendCBSWritePSRecordSource psRecordSource, long cbsStmtId) {

		boolean valid = true;
		boolean multisiteflag = psRecordSource.isMultisiteflag();

		// Build the report value map.
		Map<String, Object> mainReportValueParametersM = new HashMap<>();

		// Declare the report data source.
		List<PdfSiteStatementCollectionBean> siteStatementCollectionBeanL;

		// Site statement Map<PSSiteStmt, List<PSDetails>>
		Map<PSSiteStmt, List<PSDetails>> siteStatementM = new LinkedHashMap<>();

		// PS
		String facilityNum = EMPTY_STRING;
		PSRecord psRecord = psRecordSource.nextPSRecord(true);
		PSSiteStmt pv = null;
		while (psRecord != null) {
			if (psRecord instanceof PSSite) {
				logger.info("PS");
				facilityNum = ((PSSite) psRecord).getFacilityNum();
				appendPSMap(mainReportValueParametersM, (PSSite) psRecord, multisiteflag);
			} else if (psRecord instanceof PSPatient) {
				logger.info("PH");
				appendPHMap(mainReportValueParametersM, (PSPatient) psRecord, multisiteflag);
				appendScanLine(mainReportValueParametersM, facilityNum);
			} else if (psRecord instanceof PSSiteStmt) {
				logger.info("PV");
				pv = (PSSiteStmt) psRecord;
				List<PSDetails> siteDetailL = new ArrayList<>();
				siteStatementM.put(pv, siteDetailL);
			} else if (psRecord instanceof PSDetails) {
				logger.info("PD");
				siteStatementM.get(pv).add((PSDetails) psRecord);
			} else {
				valid = false;
				logger.error("Unknown Record Type");
			}

			if (valid) {
				psRecord = psRecordSource.nextPSRecord(true);
			} else {
				break;
			}
		}

		// TODO, If not valid log error?
		if (valid) {
			siteStatementCollectionBeanL = buildSiteStatementCollectionBean(siteStatementM, cbsStmtId, multisiteflag);

			populatePseudoStaticData(mainReportValueParametersM, facilityNum);

			return new PdfConsolidatedStatementDataBean(mainReportValueParametersM, siteStatementCollectionBeanL);
		}

		// TODO, ensure whatever calls this doesn't crash with null.
		return null;
	}

	/**
	 * Populate parameter value map for the PDF with PSSite attributes.
	 * 
	 * @param reportValueM
	 *            The parameter map to populate.
	 * @param site
	 *            The PSSite object containing the data for the site.
	 */
	private void appendPSMap(Map<String, Object> reportValueM, final PSSite site, boolean multisiteflag) {

		final Date statementDate = site.getStatementDate();
		final SimpleDateFormat statementDateFormat = new SimpleDateFormat(REPORT_DATE_FORMAT);
		reportValueM.put(STATEMENT_DATE, statementDateFormat.format(statementDate));

		// The balance determines payment date.
		final double balance = site.getStatementVal().getDouble();

		// TODO, When is this?
		final Date processDate = site.getProcessDate();
		final SimpleDateFormat processDateFormat = new SimpleDateFormat(REPORT_DATE_FORMAT);
		reportValueM.put(PAYMENT_DATE, processDateFormat.format(processDate));

		// Build the payment parameters.
		if (balance > 0) {

			// Payment Due Date is the statementDate plus FULL_PAYMENT_DAY_NUMBER_DELTA.
			// TODO, test this with roll-over from December to January.
			final SimpleDateFormat paymentDueDateFormat = new SimpleDateFormat(REPORT_DATE_FORMAT);
			Calendar c = Calendar.getInstance();
			c.setTime(statementDate);
			c.add(Calendar.DATE, FULL_PAYMENT_DAY_NUMBER_DELTA);
			final Date paymentDueDate = c.getTime();
			//Added 8/29
			
			reportValueM.put(PAYMENT_DUE_DATE, paymentDueDateFormat.format(paymentDueDate));
			//added 11/29 for single or multiple site
			if(multisiteflag){
				reportValueM.put(PAYMENT_DUE_MESSAGE, STRING_SPACE);
				reportValueM.put(BALANCE, STRING_SPACE);

			}else{
			reportValueM.put(PAYMENT_DUE_MESSAGE, PAYMENT_DUE_MESSAGE_TEXT
					+ paymentDueDateFormat.format(paymentDueDate));
			
			reportValueM.put(BALANCE, site.getStatementVal().getDoubleAsAbsoluteString());
			
			}
			reportValueM.put(AMOUNT_DUE, site.getStatementVal().getDoubleAsAbsoluteString());

		} else {

			final String paymentDueDate = NO_PAYMENT_DUE_MESSAGE_TEXT;
			reportValueM.put(PAYMENT_DUE_DATE, paymentDueDate);
			//Added 8/25/2017
			if(multisiteflag){
			reportValueM.put(PAYMENT_DUE_MESSAGE, STRING_SPACE);
			reportValueM.put(BALANCE, STRING_SPACE);
			}else{
				reportValueM.put(PAYMENT_DUE_MESSAGE, paymentDueDate);
				reportValueM.put(BALANCE, identifyValueAsCredit(site.getStatementVal().getDoubleAsAbsoluteString()));
			}
			reportValueM.put(AMOUNT_DUE, Money.DOUBLE_STRING_FORMAT);
		}
	}

	/**
	 * Populate parameter value map for the PDF with PSPatient attributes.
	 * 
	 * @param reportValueM
	 *            The parameter map to populate.
	 * @param patient
	 *            The PSPatient object containing the data for the site.
	 */
	private void appendPHMap(Map<String, Object> reportValueM, final PSPatient patient,boolean multisiteflag) {

		// Build the patient name and while doing so do not include any empty or
		// null names.
		StringBuilder patientName = new StringBuilder();
		if ((patient.getPatientFirstName() != null) && !patient.getPatientFirstName().trim().isEmpty()) {
			patientName.append(patient.getPatientFirstName().trim());
		}
		if ((patient.getPatientMiddleName() != null) && !patient.getPatientMiddleName().trim().isEmpty()) {
			patientName.append(STRING_SPACE);
			patientName.append(patient.getPatientMiddleName().trim());
		}
		if ((patient.getPatientLastName() != null) && !patient.getPatientLastName().trim().isEmpty()) {
			patientName.append(STRING_SPACE);
			patientName.append(patient.getPatientLastName().trim());
		}
		reportValueM.put(PATIENT_NAME, patientName.toString());

		// This is the patient address in the title and coupon band.
		StringBuilder address = new StringBuilder();
		if ((patient.getAddress1() != null) && !patient.getAddress1().isEmpty()) {
			address.append(patient.getAddress1());
		}
		if ((patient.getAddress2() != null) && !patient.getAddress2().isEmpty()) {
			address.append(HTML_LINE_BREAK);
			address.append(patient.getAddress2());
		}
		if ((patient.getAddress3() != null) && !patient.getAddress3().isEmpty()) {
			address.append(HTML_LINE_BREAK);
			address.append(patient.getAddress3());
		}
		final String patientCityStateZip = patient.getCity() + STRING_SPACE + patient.getState() + STRING_SPACE + patient.getZipCode();
		address.append(HTML_LINE_BREAK);
		address.append(patientCityStateZip);
		reportValueM.put(PATIENT_ADDRESS, address.toString());

		// Build the payment parameters.
		if (patient.getPrevBalance().getDouble() > 0) {
			reportValueM.put(PREVIOUS_BALANCE, patient.getPrevBalance().getDoubleAsAbsoluteString());
		} else {
			reportValueM.put(PREVIOUS_BALANCE, identifyValueAsCredit(patient.getPrevBalance().getDoubleAsAbsoluteString()));
		}
		reportValueM.put(NEW_CHARGES, patient.getTotalCharges().getDoubleAsAbsoluteString());
		reportValueM.put(PAYMENTS_RECEIVED, patient.getTotalCredits().getDoubleWithSignSuffixAsString());

		// Fix for Defect 616442
		// - Reformat Account Number
		// Yiping Yao - 01/30/2018
		String accountNumberDisp = buildAccountNumberDisp(patient);

		// Save the raw account number for later use by the Scan Line.
		// The formated account number is the one used in the actual form.
		reportValueM.put(ACCOUNT_NUMBER_RAW, accountNumberDisp);
		reportValueM.put(ACCOUNT_NUMBER, formatAccountNumber(accountNumberDisp));
		//reportValueM.put("AccountNumberRaw", patient.getAcntNumDisp().trim());
		//reportValueM.put("AccountNumber", formatAccountNumber(patient.getAcntNumDisp().trim()));

		// TODO, Figure out if this is late statement message or special notes
		// or both?
		// ADDED 8/25/2017
		StringBuilder localMessage = new StringBuilder();

		localMessage.append(HTML_NON_BREAKING_SPACE);
		localMessage.append(LOCAL_MESSAGE_TEXT);

		if(multisiteflag){
			localMessage.append(LOCAL_MESSAGE_MULTISITE_TEXT);
		}

		reportValueM.put(LOCAL_MESSAGE, localMessage.toString());
	}

	/**
	 * Create a scan line and append to the parameter map.
	 * 
	 * @param reportValueM
	 *            The parameter map to populate.
	 * @param facilityNum
	 *            The primary site/facility for the report.
	 */
	private void appendScanLine(Map<String, Object> reportValueM, final String facilityNum) {

		final String accountNumber = (String) reportValueM.get(ACCOUNT_NUMBER_RAW);
		final String amountDue = (String) reportValueM.get(AMOUNT_DUE);

		final ScanLine scanLine = new ScanLine(facilityNum, accountNumber, amountDue);

		// Append these values.
		reportValueM.put(MAILING_LABEL, scanLine.getCalculatedScanLine());
	}

	/**
	 * Populate pseudo static data in the parameter map.
	 * 
	 * @param reportValueM
	 *            The parameter map to populate.
	 * @param facilityNum
	 *            The primary site for which to use as lookup key for the pseudo
	 *            static data.
	 */
	private void populatePseudoStaticData(Map<String, Object> reportValueM, final String facilityNum) {

		StationInfo station = StationInfoUtil.getStation(facilityNum);
		reportValueM.put(SITE_LOCATION_NAME, station.getFacilityDesc());

		// This is the address at the top of the report header and is the VAMC's address.
		Address stationAddress = station.getAddress();
		StringBuilder address = new StringBuilder();
		if ((stationAddress.getAddress1() != null) && !stationAddress.getAddress1().isEmpty()) {
			address.append(stationAddress.getAddress1());
		}
		if ((stationAddress.getAddress2() != null) && !stationAddress.getAddress2().isEmpty()) {
			address.append(HTML_LINE_BREAK);
			address.append(stationAddress.getAddress2());
		}
		if ((stationAddress.getAddress3() != null) && !stationAddress.getAddress3().isEmpty()) {
			address.append(HTML_LINE_BREAK);
			address.append(stationAddress.getAddress3());
		}

		// This is wrong, and station.getZipCode() returns a ZipCode object, not actual ZIP code in string format.
		// And it is not needed, since the SITE_CITY_STATE_ZIP already supplies the Cite, State and ZIP Code.
		// This was not shown since HTML only has one line for the address.
		// Yiping Yao - 02/21/2018
		//final String siteCityStateZip = station.getCity() + STRING_SPACE + station.getState() + STRING_SPACE + station.getZipCode();
		//address.append(HTML_LINE_BREAK);
		//address.append(siteCityStateZip);

		reportValueM.put(SITE_ADDRESS, address.toString());

		// Fix for Defect 680703
		// - Format ZIP Code as: #####-####
		// Yiping Yao - 02/21/2018
		final String plus4Code = ( station.getZipCode().getZipCode2() == null ||
								   station.getZipCode().getZipCode2().trim().isEmpty() ||
								   station.getZipCode().getZipCode2().equals("0000") ) ?
								   EMPTY_STRING : STRING_DASH + station.getZipCode().getZipCode2();
		reportValueM.put(SITE_CITY_STATE_ZIP,
						 String.format("%s, %s %s", station.getCity(), station.getState(),
								 	   station.getZipCode().getZipCode1() + plus4Code));
		//reportValueM.put(SITE_CITY_STATE_ZIP,
		//		String.format("%s, %s %s", station.getCity(), station.getState(), station.getZipCode().getZipCode1()));

		// This is the phone number of the HRC Call Center (per VISN).
		reportValueM.put(QUESTIONS_PHONE, station.getTelephoneNum1());

		// For some reason this is hard coded on the form.
		// So it is hard coded here for easier update.
		reportValueM.put(PAY_BY_PHONE, PAY_BY_PHONE_TEXT);
		reportValueM.put(PAY_HOURS, PAY_HOURS_TEXT);
		
		// MAIL TO: mailing address.
		// One address per CPAC.
		// TODO, change this to html field type and concatenate a single field.
		reportValueM.put(MAIL_TO_ADDRESS, MAIL_TO_ADDRESS_TEXT);
	}

	/**
	 * Build a PdfSiteStatementCollectionBean from the site statement map to be
	 * used as a Jasper data source.
	 * 
	 * @param siteStatementM
	 *            The site statement map.
	 * @param cbsStmtId
	 * @param isMultiSite - the multi-site flag
	 * @return List of PdfSiteStatementCollectionBean. Note that the list is
	 *         size of one because the Jasper data source that is constructed
	 *         requires a list.
	 */
	private List<PdfSiteStatementCollectionBean> buildSiteStatementCollectionBean(
			Map<PSSiteStmt, List<PSDetails>> siteStatementM, long cbsStmtId, boolean isMultiSite) {

		PdfSiteStatementCollectionBean siteStatementCollection = new PdfSiteStatementCollectionBean();

		List<PdfSiteStatementBean> siteStatementBeanL = new ArrayList<>();

		for (Map.Entry<PSSiteStmt, List<PSDetails>> siteEntry : siteStatementM.entrySet()) {

			PdfSiteStatementBean site = new PdfSiteStatementBean();

			// Fix for Defect 669283
			// - Remove Facility Title for single site
			// Yiping Yao - 02/01/2018
			if (isMultiSite)
			{
				final String facilityTitle = StationInfoUtil.getStation(siteEntry.getKey().getFacilityNum()).getFacilityDesc();

				// Fix for Defect 661784
				// - Remove # Facility Number
				// Yiping Yao - 01/22/2018
				site.setFacility(facilityTitle);
				//site.setFacility(facilityTitle + " # " + siteEntry.getKey().getFacilityNum());
			}

			//set late statement flag for site if statement is marked late
			final CBSStmt cbsStmt = this.cbsStmtDAO.get(cbsStmtId);

			if(cbsStmt.getLateStmtMsg()){
				site.setLateStmtFlag(Boolean.TRUE);
			}

			List<PdfSiteStatementEntryBean> pdfSiteStatementEntryBeanL = new ArrayList<>();

			boolean isAddingSpace = false;

			for (PSDetails details : siteEntry.getValue()) {
				// TODO, split the detail into multiple lines.
				// This is necessary so line calculation is correct for populating the form with the specified number of lines.
				// TODO, check trans desc and wrap with [] for suspended charges

				// Fix for Defect 630421
				// - Not display "0.00" in amount
				// Yiping Yao - 01/11/2018
				String transAmount = details.getTransAmount().getDoubleWithSignSuffixAsString();

				transAmount = (transAmount.equals(Money.DOUBLE_STRING_FORMAT)) ? STRING_SPACE : transAmount;

				// Fix for Defect 661797
				// - line up decimals in amount column
				//   Add space to positive number (without the minus sign "-"),
				//   so that the cents are line up correctly.
				// Yiping Yao - 01/22/2018

				if (transAmount.length() > 1 && transAmount.charAt(transAmount.length() - 1) != '-')
				{
					isAddingSpace = true;
					transAmount += HTML_NON_BREAKING_SPACE;
				}

				PdfSiteStatementEntryBean detailEntry = new PdfSiteStatementEntryBean(formatDetailDescription(details.getTransDesc(), details.getDatePosted()),
						transAmount, details.getReferenceNum());
				pdfSiteStatementEntryBeanL.add(detailEntry);
			}

			site.setRecords(pdfSiteStatementEntryBeanL);

			// Fix for Defects 630437 and 630398
			// - Add text to facility total label for multi-site
			// - Remove facility total for single site
			// Yiping Yao - 01/18/2018
			//site.setTotalLabel("FACILITY #" + siteEntry.getKey().getFacilityNum() + " TOTAL");
			if (isMultiSite)
			{
				site.setTotalLabel("PREVIOUS AND CURRENT CHARGES FOR FACILITY #" + siteEntry.getKey().getFacilityNum() + " TOTAL");

				// Fix for Defect 661797
				// - line up decimals in total column as well
				//   Add space to total if there is negative amount,
				//   so that the cents are line up correctly.
				// Yiping Yao - 01/22/2018
				String siteTotal = siteEntry.getKey().getStationTotalAmount().getDoubleAsString();

				if (isAddingSpace)
				{
					siteTotal += HTML_NON_BREAKING_SPACE;
				}

				site.setTotal(siteTotal);
			}

			siteStatementBeanL.add(site);
		}

		siteStatementCollection.setSiteStatementL(siteStatementBeanL);

		// Fix for Defect 711459
		// - This may fix the missing details line on the second page by subtract 1.
		// Yiping Yao - 04/04/2018
		// Split the site statement collection bean.
		siteStatementCollection.split(MAX_LENGTH_FIRST_PAGE - 1);

		// Add filler space to extend box to footer of page
		siteStatementCollection.addFillerSpace(MAX_LENGTH_FIRST_PAGE, MAX_LENGTH_EXTENSION_PAGE);

		return Arrays.asList(siteStatementCollection);
	}

	/**
	 * Format the description text as shown on the examples
	 * provided by AITC.
	 * 
	 * @param detailDesc
	 *            ----The original description text.
	 * @return The formatted description.
	 */
	private String formatDetailDescription(String detailDesc, Date datePosted) {
		if (detailDesc == null) {
			return EMPTY_STRING;
		}

		String parsedDesc = detailDesc;

		parsedDesc = parsedDesc.replace(" FD:", " Fill Date: ");
		parsedDesc = parsedDesc.replace(" RX:", " RX#");
		parsedDesc = parsedDesc.replace("(NSC)", EMPTY_STRING);

		// Fix for Defect 680682
		// - Use the actual payment posted date, replacing only "PAYMENT (MM/dd/yyyy)"
		// Yiping Yao - 02/23/2018
		//
		// Fix for Defect 711459
		// - Replace "PAYMENT" with "PAYMENT POSTED ON" followed by the actual payment posted date,
		//   and remove / ignore everything else after the original "PAYMENT".
		// Yiping Yao - 04/04/2018
		if ( datePosted != null && detailDesc.startsWith("PAYMENT") )
		//	 ( detailDesc.startsWith("PAYMENT") &&
		//	   detailDesc.contains("(") &&
		//	   detailDesc.contains("/") &&
		//	   detailDesc.contains("/20") &&
		//	   detailDesc.contains(")") ) )
		{
			final SimpleDateFormat postedDateFormat = new SimpleDateFormat(REPORT_DATE_FORMAT);
			parsedDesc = "PAYMENT POSTED ON" + STRING_SPACE + postedDateFormat.format(datePosted);
		}
		//parsedDesc = parsedDesc.replace("PAYMENT", "PAYMENT POSTED ON");

		return parsedDesc.trim();
	}

	/**
	 * Format the account number for easier readability as shown on the examples
	 * provided by AITC.
	 * 
	 * @param accountNumber
	 *            The account number.
	 * @return The formatted account number.
	 * @throws IndexOutOfBoundsException
	 *             If an error during formatting an exception is thrown.
	 */
	public static String formatAccountNumber(final String accountNumber) throws IndexOutOfBoundsException {

		// The format of the account number displayed on the report appears to
		// be:
		// 3 character site number + space + 3 character + space + 3 character +
		// space + 4 character + space + remaining character
		StringBuilder formattedAccountNumber = new StringBuilder();

		formattedAccountNumber.append(substringByPositionLength(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_1,
				ACCOUNT_NUMBER_SPLIT_LENGTH_1, true));

		formattedAccountNumber.append(STRING_SPACE);
		formattedAccountNumber.append(substringByPositionLength(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_2,
				ACCOUNT_NUMBER_SPLIT_LENGTH_2, true));

		formattedAccountNumber.append(STRING_SPACE);
		formattedAccountNumber.append(substringByPositionLength(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_3,
				ACCOUNT_NUMBER_SPLIT_LENGTH_3, true));

		formattedAccountNumber.append(STRING_SPACE);
		formattedAccountNumber.append(substringByPositionLength(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_4,
				ACCOUNT_NUMBER_SPLIT_LENGTH_4, true));

		formattedAccountNumber.append(STRING_SPACE);
		// Fix for Defect 616442
		// - Reformat Account Number
		// Yiping Yao - 01/30/2018
		formattedAccountNumber.append(substringByPositionLength(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_5,
				ACCOUNT_NUMBER_SPLIT_LENGTH_5, false));
		//formattedAccountNumber.append(substringByPosition(accountNumber, ACCOUNT_NUMBER_SPLIT_POSITION_5));

		return formattedAccountNumber.toString();
	}

	/**
	 * Get a substring of the input string by the specified position.
	 * 
	 * @param input
	 *            The input string.
	 * @param position
	 *            The start position for the substring.
	 * @return The substring.
	 * @throws IndexOutOfBoundsException
	 *             If an error during formatting an exception is thrown.
	 */
	/*
	private static String substringByPosition(final String input, final int position) throws IndexOutOfBoundsException {

		// Verify the position is valid for the input.
		if ((input.length() - 1) < position) {
			throw new IndexOutOfBoundsException("Input (" + input + ") with position (" + position + ") is invalid");
		}

		// Get the substring.
		final String ss = input.substring(position);

		return ss;
	}
	*/

	/**
	 * Get a substring of the input string by the specified position and the
	 * desired length.
	 * 
	 * @param input
	 *            The input string.
	 * @param position
	 *            The start position for the substring
	 * @param length
	 *            The length of the substring.
	 * @param checkLength
	 *            Flag to indicate if the result should be checked for an error.
	 * @return The substring.
	 * @throws IndexOutOfBoundsException
	 *             If an error during formatting an exception is thrown.
	 */
	private static String substringByPositionLength(final String input, final int position, final int length,
			final boolean checkLength) throws IndexOutOfBoundsException {

		// Verify the position is valid for the input.
		if ((input.length() - 1) < position) {
			throw new IndexOutOfBoundsException("Input (" + input + ") with position (" + position + ") is invalid");
		}

		// Optionally, verify the input will support the substring.
		if (checkLength && ((position + length) > input.length())) {
			throw new IndexOutOfBoundsException("Input (" + input + ") will not support specified position (" + position
					+ ") and length (" + length + ") substring");
		}

		// Get the substring.
		final String ss = input.substring(position, Math.min(position + length, input.length()));

		// If necessary, check the length and if not the expected length it is
		// invalid.
		if (checkLength && (ss.length() != length)) {
			throw new IndexOutOfBoundsException("Expected output (" + ss + ") with length (" + length + ")");
		}

		return ss;
	}

	/**
	 * Create a string that makes the value represented as a credit.
	 * @param value The value to convert.
	 * @return The credit representation of the value.
	 */
	private static String identifyValueAsCredit(final String value) {
		return value + CREDIT_SUFFIX;
	}

	// Fix for Defect 616442
	// - Reformat Account Number
	// Yiping Yao - 01/30/2018
	/**
	 * Construct Account Number:
	 * 
	 *  Facility Number (3 digits) +
	 *  DFN (13 digits, padded with zero) +
	 *  Patient’s Last Name (5 characters, padded with spaces)
	 * 
	 */
	private String buildAccountNumberDisp(PSPatient patient)
	{
		if (patient == null)
		{
			return EMPTY_STRING;
		}

		String acntNumDisp = patient.getAcntNumDisp();

		// The first 3 digits / characters in the original acntNumDisp are Facility Number
		String facilityNumber = acntNumDisp.substring(ACCOUNT_NUMBER_SPLIT_POSITION_1, ACCOUNT_NUMBER_SPLIT_LENGTH_1);

		// The first 3 digits in DFN number are facility number, and the rest is the DFN
		String dfnNumber = String.valueOf(patient.getDfnNumber()).substring(ACCOUNT_NUMBER_SPLIT_LENGTH_1);

		// The original acntNumDisp = facility number (3) + account number (10) + last name (5)
		String lastName = acntNumDisp.substring(13);

		return String.format("%-3.3s%013d%-5.5s", facilityNumber, Long.valueOf(dfnNumber), lastName).trim();
	}
}
