package gov.va.cpss.email.impl;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;

import gov.va.cpss.email.SmtpMailConfigurer;

/**
 * Implementation of SmtpMailConfigurer. Ensures the mail properties are set as
 * system properties on initialization. Properties are set in order of
 * precedence as follows:
 * 
 * 1. Application Properties (i.e. application.properties file),
 * 2. System Properties (i.e. -D command line parameters),
 * 3. Default values: localhost,
 * 100 for mail.smtp.host and mail.smtp.port, respectively.
 * 
 * @author Brad Pickle
 *
 */
public class DefaultSmtpMailConfigurerImpl implements SmtpMailConfigurer, InitializingBean {

	private final Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

	private String mailSmtpHostApplicationProperty;

	private String mailSmtpPortApplicationProperty;

	public String getMailSmtpHostApplicationProperty() {
		return mailSmtpHostApplicationProperty;
	}

	public void setMailSmtpHostApplicationProperty(String mailSmtpHostApplicationProperty) {
		this.mailSmtpHostApplicationProperty = mailSmtpHostApplicationProperty;
	}

	public String getMailSmtpPortApplicationProperty() {
		return mailSmtpPortApplicationProperty;
	}

	public void setMailSmtpPortApplicationProperty(String mailSmtpPortApplicationProperty) {
		this.mailSmtpPortApplicationProperty = mailSmtpPortApplicationProperty;
	}

	@Override
	public String getMailSmtpHost() {
		return System.getProperty(MAIL_SMTP_HOST_PROPERTY);
	}

	@Override
	public void setMailSmtpHost(String mailSmtpHost) {
		if ((mailSmtpHost == null) || mailSmtpHost.trim().isEmpty()) {
			System.clearProperty(MAIL_SMTP_HOST_PROPERTY);
		} else {
			System.setProperty(MAIL_SMTP_HOST_PROPERTY, mailSmtpHost);
		}
	}

	@Override
	public String getMailSmtpPort() {
		return System.getProperty(MAIL_SMTP_PORT_PROPERTY);
	}

	@Override
	public void setMailSmtpPort(String mailSmtpPort) {
		if ((mailSmtpPort == null) || mailSmtpPort.trim().isEmpty()) {
			System.clearProperty(MAIL_SMTP_PORT_PROPERTY);
		} else {
			System.setProperty(MAIL_SMTP_PORT_PROPERTY, mailSmtpPort);
		}
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		setPropertyWithDefault(MAIL_SMTP_HOST_PROPERTY, mailSmtpHostApplicationProperty, DEFAULT_MAIL_SMTP_HOST);
		setPropertyWithDefault(MAIL_SMTP_PORT_PROPERTY, mailSmtpPortApplicationProperty, "" + DEFAULT_MAIL_SMTP_PORT);
		logger.info("Configured MAIL_SMTP_HOST_PROPERTY:" + getMailSmtpHost());
		logger.info("Configured MAIL_SMTP_PORT_PROPERTY:" + getMailSmtpPort());
	}

	private void setPropertyWithDefault(String propertyName, String applicationPropertyValue,
			String defaultPropertyValue) {
		if ((applicationPropertyValue == null) || applicationPropertyValue.trim().isEmpty()) {
			final String currentPropertyValue = System.getProperty(propertyName);
			if ((currentPropertyValue == null) || currentPropertyValue.trim().isEmpty())
				System.setProperty(propertyName, defaultPropertyValue);
		} else {
			System.setProperty(propertyName, applicationPropertyValue);
		}
	}
}
