package gov.va.cpss.dao.impl;

import static gov.va.cpss.ESAPI.EsapiValidationType.LOG_FORGING;
import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import gov.va.cpss.dao.ProcessStatusDAO;
import gov.va.cpss.model.ProcessStatus;

/**
 * An implementation of the ProcessStatusDAO interface.
 */
public class ProcessStatusDAOImpl implements ProcessStatusDAO {

	private final Logger daoLogger;

	private JdbcTemplate jdbcTemplate;

	private Map<Integer, ProcessStatus> idToTypeM;
	private Map<ProcessStatus.Status, Integer> enumToIdM;

	public ProcessStatusDAOImpl() {

		daoLogger = Logger.getLogger(this.getClass().getCanonicalName());

		idToTypeM = new HashMap<>();
		enumToIdM = new HashMap<>();
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	@Override
	public void initTypeMaps() {

		List<ProcessStatus> bSL = statusTypeList();

		for (ProcessStatus bS : bSL) {

			ProcessStatus.Status enumValue = ProcessStatus.Status.lookupByName(bS.getName());
			if (enumValue == null) {
				daoLogger
						.error("Unrecognized Process Status in database: " + validateStringInput(bS.getName(), LOG_FORGING));
			} else {
				bS.setStatus(enumValue);
			}
			idToTypeM.put(bS.getId(), bS);
			enumToIdM.put(enumValue, bS.getId());
		}
	}

	@Override
	public List<ProcessStatus> statusTypeList() {
		String sql = "SELECT * FROM ProcStatus";
		List<ProcessStatus> statusL = jdbcTemplate.query(sql, new RowMapper<ProcessStatus>() {

			@Override
			public ProcessStatus mapRow(ResultSet rs, int rowNum) throws SQLException {
				ProcessStatus statusType = new ProcessStatus();

				statusType.setId(rs.getInt("id"));
				statusType.setName(rs.getString("name"));
				statusType.setDescription(rs.getString("description"));

				return statusType;
			}

		});

		return statusL;
	}

	@Override
	public ProcessStatus getStatusType(int id) {
		return idToTypeM.get(id);
	}

	@Override
	public Integer getStatusFromEnum(ProcessStatus.Status enumValue) {
		return enumToIdM.get(enumValue);
	}

}
