package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.PSSiteDAO;
import gov.va.cpss.model.fps.PSSite;

/**
 * An implementation of the PSSiteDAO interface.
 * 
 * Added getAllForPSReceivedStatusId() implementation and refactored RowMapper into static class. Brad - 12/19/2016
 */
public class PSSiteDAOImpl implements PSSiteDAO {

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public PSSiteDAOImpl() {
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public Long getLastIndex() {

		// query
		final String sql = "SELECT * FROM (SELECT * FROM PSSite ORDER BY id DESC) WHERE rownum <= 1";

		List<Long> results = jdbcTemplate.query(sql, new RowMapper<Long>() {
			@Override
			public Long mapRow(ResultSet rs, int row) throws SQLException {
				return new Long(rs.getLong(1));
			}
		});

		if ((results != null) && !results.isEmpty()) {
			return results.get(0);
		}

		return null;
	}

	@Override
	public void deleteByReceivedId(final long receivedId) {

		final String sql = "DELETE FROM PSSite where psReceived_Id = ?";
		jdbcTemplate.update(sql, receivedId);
	}

	@Override
	public PSSite getById(final long id) {
		// query
		final String sql = "SELECT * FROM PSSite WHERE id = " + id;

		return jdbcTemplate.query(sql, new ResultSetExtractor<PSSite>() {

			@Override
			public PSSite extractData(ResultSet rs) throws SQLException, DataAccessException {
				if (rs.next()) {
					PSSite psSite = new PSSite(rs.getLong("id"));
					psSite.setPsReceivedId(rs.getLong("psReceived_Id"));
					psSite.setSeqNum(rs.getInt("seqNum"));
					psSite.setTotSeqNum(rs.getInt("totSeqNum"));
					psSite.setFacilityNum(rs.getString("facilityNum"));
					psSite.setFacilityPhoneNum(rs.getString("facilityPhoneNum"));
					psSite.setTotStatement(rs.getInt("totStatement"));
					psSite.setStatementVal(new Money(rs.getString("statementValRaw")));
					psSite.setStatementDate(rs.getTimestamp("statementDate"));
					psSite.setProcessDate(rs.getTimestamp("processDate"));
					psSite.setCreatedBy(rs.getString("createdBy"));
					psSite.setCreatedDate(new java.util.Date(rs.getDate("createdDate").getTime()));
					psSite.setModifiedBy(rs.getString("modifiedBy"));
					psSite.setModifiedDate(new java.util.Date(rs.getDate("modifiedDate").getTime()));
					return psSite;
				}

				return null;
			}

		});
	}

	@Override
	public List<PSSite> getAllForPSReceivedStatusId(final int statusId) {
		String sql = "SELECT * FROM PSSite INNER JOIN PSReceived ON PSSite.psReceived_Id = PSReceived.id "
				+ "WHERE PSReceived.status_Id = " + statusId;

		List<PSSite> statusL = jdbcTemplate.query(sql, new PSSiteRowMapper());

		return statusL;
	}

	@Override
	public List<PSSite> getAllByPSReceivedID(final long receivedId) {

		String sql = "SELECT * FROM PSSite WHERE psReceived_Id = " + receivedId + " order by id";
		List<PSSite> statusL = jdbcTemplate.query(sql, new PSSiteRowMapper());

		return statusL;
	}

	/**
	 * Maps a ResultSet to a new CBSStmt object.
	 * 
	 * @author Brad Pickle
	 *
	 */
	private static class PSSiteRowMapper implements RowMapper<PSSite> {

		@Override
		public PSSite mapRow(ResultSet rs, int rowNum) throws SQLException {
			PSSite psSite = new PSSite(rs.getLong("id"));
			psSite.setPsReceivedId(rs.getLong("psReceived_Id"));
			psSite.setSeqNum(rs.getInt("seqNum"));
			psSite.setTotSeqNum(rs.getInt("totSeqNum"));
			psSite.setFacilityNum(rs.getString("facilityNum"));
			psSite.setFacilityPhoneNum(rs.getString("facilityPhoneNum"));
			psSite.setTotStatement(rs.getInt("totStatement"));
			psSite.setStatementVal(new Money(rs.getString("statementValRaw")));
			psSite.setStatementDate(rs.getTimestamp("statementDate"));
			psSite.setProcessDate(rs.getTimestamp("processDate"));
			psSite.setCreatedBy(rs.getString("createdBy"));
			psSite.setCreatedDate(new java.util.Date(rs.getDate("createdDate").getTime()));
			psSite.setModifiedBy(rs.getString("modifiedBy"));
			psSite.setModifiedDate(new java.util.Date(rs.getDate("modifiedDate").getTime()));

			return psSite;
		}

	}
	
	@Override
	public List<Long> getAllIDByPSReceivedID(final long receivedId) {

		String sql = "SELECT id FROM PSSite WHERE psReceived_Id = ? order by id";
		List<Long> results = jdbcTemplate.query(sql, new PreparedStatementSetter() {
			public void setValues(PreparedStatement preparedStatement) throws SQLException {
				preparedStatement.setLong(1, receivedId);
			}
		}, new RowMapper<Long>() {
			@Override
			public Long mapRow(ResultSet rs, int row) throws SQLException {
				return new Long(rs.getLong(1));
			}
		});

		return results;
	}

	@Override
	public long save(PSSite site) {
		// insert
		final String sql = "INSERT INTO PSSite (psReceived_Id, seqNum, totSeqNum, facilityNum, facilityPhoneNum, totStatement, statementVal, statementValRaw, statementDate, processDate) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setLong(1, site.getPsReceivedId());
				ps.setInt(2, site.getSeqNum());
				ps.setInt(3, site.getTotSeqNum());
				ps.setString(4, site.getFacilityNum());
				ps.setString(5, site.getFacilityPhoneNum());
				ps.setInt(6, site.getTotStatement());
				ps.setDouble(7, site.getStatementVal().getDouble());
				ps.setString(8, site.getStatementVal().getCobol());
				ps.setDate(9, new Date(site.getStatementDate().getTime()));
				ps.setDate(10, new Date(site.getProcessDate().getTime()));
				return ps;
			}
		}, keyHolder);
		site.setId(keyHolder.getKey().longValue());

		// Load the new audit fields and set on statement
		cbssAuditDatesDAO.getAuditDates("PSSite", site.getId(), site);

		return site.getId();
	}

}
