package gov.va.cpss.dao.impl;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.PreparedStatementSetter;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.cobol.Money;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.dao.CBSStmtDAO;
import gov.va.cpss.dao.util.SQLValuesList;
import gov.va.cpss.model.BooleanChar;
import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.ADFileRecord;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * An implementation of the CBSStmtDAO interface.
 */
public class CBSStmtDAOImpl implements CBSStmtDAO {

	public final static int BATCH_SELECT_MAX_SIZE = 100;

	private static final String CBSSTMT_SELECT_FIELDS = "a.id AS id, a.replacedStmtId AS replacedStmtId, a.messageId AS messageId, "
			+ "a.accountId AS accountId, a.acntNumDisp AS acntNumDisp, a.amntDue AS amntDue, a.preBalance AS preBalance, "
			+ "a.totalCharges AS totalCharges, a.totalCredits AS totalCredits, a.newBalance AS newBalance, a.lateStmtMsg AS lateStmtMsg, "
			+ "a.statementDate AS statementDate, a.processDate AS processDate, a.batchRunId AS batchRunId, a.statusId AS statusId, "
			+ "a.printAckId AS printAckId, a.printDate AS printDate, "
			+ "a.createdBy AS createdBy, a.createdDate AS createdDate, a.modifiedBy AS modifiedBy, a.modifiedDate AS modifiedDate ";

	private final Logger daoLogger;

	private JdbcTemplate jdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSStmtDAOImpl() {
		daoLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public long save(CBSStmt statement) {
		// insert
		final String sql = "INSERT INTO CBSStmt (accountId, acntNumDisp, amntDue, preBalance, totalCharges, totalCredits, newBalance, lateStmtMsg, statementDate, processDate, batchRunId, statusId, replacedStmtId) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		KeyHolder keyHolder = new GeneratedKeyHolder();

		jdbcTemplate.update(new PreparedStatementCreator() {
			@Override
			public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
				PreparedStatement ps = connection.prepareStatement(sql, new String[] { "id" });
				ps.setLong(1, statement.getAccountId());
				ps.setString(2, statement.getAcntNumDisp());
				ps.setDouble(3, statement.getAmountDue().getDouble());
				ps.setDouble(4, statement.getPrevBalance().getDouble());
				ps.setDouble(5, statement.getTotalCharges().getDouble());
				ps.setDouble(6, statement.getTotalCredits().getDouble());
				ps.setDouble(7, statement.getNewBalance().getDouble());
				ps.setString(8, Character.toString(BooleanChar.from(statement.getLateStmtMsg()).getCharValue()));
				ps.setDate(9, new Date(statement.getStatementDate().getTime()));
				ps.setDate(10, new Date(statement.getProcessDate().getTime()));
				ps.setLong(11, statement.getBatchRunId());
				ps.setInt(12, statement.getStatusId());
				ps.setLong(13, statement.getReplacedStmtId());
				return ps;
			}
		}, keyHolder);
		statement.setId(keyHolder.getKey().longValue());

		// Load the new audit fields and set on statement
		cbssAuditDatesDAO.getAuditDates("CBSStmt", statement.getId(), statement);

		return statement.getId();
	}

	@Override
	public void saveBatch(List<CBSStmt> statements) {
		final String sql = "INSERT INTO CBSStmt (accountId, acntNumDisp, amntDue, preBalance, totalCharges, "
				+ "totalCredits, newBalance, lateStmtMsg, statementDate, processDate, batchRunId, statusId, "
				+ "replacedStmtId) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

		jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				CBSStmt statement = statements.get(i);

				ps.setLong(1, statement.getAccountId());
				ps.setString(2, statement.getAcntNumDisp());
				ps.setDouble(3, statement.getAmountDue().getDouble());
				ps.setDouble(4, statement.getPrevBalance().getDouble());
				ps.setDouble(5, statement.getTotalCharges().getDouble());
				ps.setDouble(6, statement.getTotalCredits().getDouble());
				ps.setDouble(7, statement.getNewBalance().getDouble());
				ps.setString(8, Character.toString(BooleanChar.from(statement.getLateStmtMsg()).getCharValue()));
				ps.setDate(9, new Date(statement.getStatementDate().getTime()));
				ps.setDate(10, new Date(statement.getProcessDate().getTime()));
				ps.setLong(11, statement.getBatchRunId());
				ps.setInt(12, statement.getStatusId());
				ps.setLong(13, statement.getReplacedStmtId());

			}

			@Override
			public int getBatchSize() {
				return statements.size();
			}
		});

	}

	@Override
	public CBSStmt getExistingStatementByAccountAndMonth(final long accountNumber, final java.util.Date statementDate) {

		daoLogger.debug("GETTING STATEMENT FOR DATE: |" + new SimpleDateFormat("MM-yyyy").format(statementDate) + "|");

		// query
		final String sql = "SELECT * FROM (SELECT * FROM CBSStmt where accountId=? and to_char(statementDate,'mm-yyyy')=? ORDER BY id DESC) WHERE rownum <= 1";

		CBSStmt results = jdbcTemplate.query(sql, new PreparedStatementSetter() {
			public void setValues(PreparedStatement preparedStatement) throws SQLException {
				preparedStatement.setLong(1, accountNumber);
				preparedStatement.setString(2, new SimpleDateFormat("MM-yyyy").format(statementDate));
			}
		}, new ResultSetExtractor<CBSStmt>() {
			@Override
			public CBSStmt extractData(ResultSet rs) throws SQLException {
				if (rs.next()) {
					CBSStmt statement = new CBSStmt(rs.getLong("batchRunId"), rs.getLong("accountId"),
							rs.getInt("statusId"));

					statement.setId(rs.getLong("id"));
					statement.setMessageId(rs.getLong("messageId"));
					statement.setAcntNumDisp(rs.getString("acntNumDisp"));
					statement.setAmountDue(new Money(rs.getDouble("amntDue"), 9));
					statement.setPrevBalance(new Money(rs.getDouble("preBalance"), 9));
					statement.setTotalCharges(new Money(rs.getDouble("totalCharges"), 9));
					statement.setTotalCredits(new Money(rs.getDouble("totalCredits"), 9));
					statement.setNewBalance(new Money(rs.getDouble("newBalance"), 9));
					statement.setLateStmtMsg(BooleanChar.from(rs.getString("lateStmtMsg")).isTrue());
					statement.setStatementDate(rs.getDate("statementDate"));
					statement.setProcessDate(rs.getDate("processDate"));
					statement.setReplacedStmtId(rs.getLong("replacedStmtId"));
					statement.setCreatedBy(rs.getString("createdBy"));
					statement.setCreatedDate(rs.getDate("createdDate"));
					statement.setModifiedBy(rs.getString("modifiedBy"));
					statement.setModifiedDate(rs.getDate("modifiedDate"));

					return statement;
				}
				return null;
			}
		});

		return results;
	}

	@Override
	public Integer getStatusIdForExistingStatementByAccountAndMonth(final long accountNumber,
			final java.util.Date statementDate) {
		daoLogger.debug("GETTING statusId FOR STATEMENT FOR DATE: |"
				+ new SimpleDateFormat("MM-yyyy").format(statementDate) + "|");

		// query
		final String sql = "SELECT statusId FROM (SELECT * FROM CBSStmt where accountId=? and to_char(statementDate,'mm-yyyy')=? ORDER BY id DESC) WHERE rownum <= 1";

		Integer results = jdbcTemplate.query(sql, new PreparedStatementSetter() {
			public void setValues(PreparedStatement preparedStatement) throws SQLException {
				preparedStatement.setLong(1, accountNumber);
				preparedStatement.setString(2, new SimpleDateFormat("MM-yyyy").format(statementDate));
			}
		}, new ResultSetExtractor<Integer>() {
			@Override
			public Integer extractData(ResultSet rs) throws SQLException {
				if (rs.next()) {
					Integer statusId = rs.getInt("statusId");

					return statusId;
				}
				return null;
			}
		});

		return results;

	}

	@Override
	public List<CBSStmt> getMessageStatementsForSite(final long cbsMessageId, final String stationNum) {
		final String sql = "SELECT " + CBSSTMT_SELECT_FIELDS
				+ "FROM CBSStmt a INNER JOIN CBSSiteStmt primarySite ON a.id = primarySite.stmtId "
				+ "WHERE a.messageId=" + cbsMessageId + " AND primarySite.isPrimary='Y' AND primarySite.stationNum='"
				+ stationNum + "' ";

		final List<CBSStmt> results = jdbcTemplate.query(sql, new CBSStmtRowMapper());

		return results;
	}

	@Override
	public Map<String, CBSStmt> getStatementsForPrintAck(final PrintAckRec printAckRec,
			final java.util.Date statementDate, final int sentStatusId, final int ackStatusId) {

		final List<ADFileRecord> printAckDetailL = printAckRec.getPrintAckDetailL();

		final Map<String, CBSStmt> cbsStmtByAccountM = new HashMap<>();

		if ((printAckDetailL != null) && (printAckDetailL.size() > 0)) {
			List<ADFileRecord> nonNullPrintAckDetailL = printAckDetailL.stream()
					.filter(adFileRecord -> adFileRecord != null).collect(Collectors.toList());

			final String formattedStatementDate = new SimpleDateFormat("MM-dd-yyyy").format(statementDate);

			final String sqlBase = "SELECT patient.oldAcntNum AS oldAcntNum, " + CBSSTMT_SELECT_FIELDS
					+ "FROM CBSStmt a INNER JOIN CBSSiteStmt siteStmt ON a.id = siteStmt.stmtId "
					+ "  INNER JOIN CBSSitePatient patient ON siteStmt.id = patient.siteStmtId "
					+ "WHERE to_char(a.statementDate,'mm-dd-yyyy')='" + formattedStatementDate + "' "
					+ " AND a.statusId IN (" + sentStatusId + ", " + ackStatusId + ") AND siteStmt.stationNum='"
					+ printAckRec.getStationNum() + "' " + " AND patient.oldAcntNum IN ";

			final StatementsForPrintAckPreparedStatementSetter pss = new StatementsForPrintAckPreparedStatementSetter(
					nonNullPrintAckDetailL, sqlBase);

			final CBSStmtRowMapper cbsStmtRowMapper = new CBSStmtRowMapper();
			final RowMapper<CBSStmtAndAccount> cbsStmtAndAccountRowMapper = new RowMapper<CBSStmtAndAccount>() {

				@Override
				public CBSStmtAndAccount mapRow(ResultSet rs, int rowNum) throws SQLException {
					CBSStmtAndAccount cbsStmtAndAccount = new CBSStmtAndAccount();

					cbsStmtAndAccount.setCbsStmt(cbsStmtRowMapper.mapRow(rs, rowNum));
					cbsStmtAndAccount.setOldAcntNum(rs.getString("oldAcntNum"));

					return cbsStmtAndAccount;
				}

			};

			while (pss.getRemainingCount() > 0) {

				List<CBSStmtAndAccount> nextCBSStmtAndAccountL = jdbcTemplate.query(pss.getNextBatchSelectSQL(), pss,
						cbsStmtAndAccountRowMapper);
				for (CBSStmtAndAccount cbsStmtAndAccount : nextCBSStmtAndAccountL) {
					cbsStmtByAccountM.put(cbsStmtAndAccount.getOldAcntNum(), cbsStmtAndAccount.getCbsStmt());
				}
			}

		}

		return cbsStmtByAccountM;
	}

	private static class CBSStmtAndAccount {
		private String oldAcntNum;
		private CBSStmt cbsStmt;
		
		public String getOldAcntNum() {
			return oldAcntNum;
		}
		public void setOldAcntNum(String oldAcntNum) {
			this.oldAcntNum = oldAcntNum;
		}
		public CBSStmt getCbsStmt() {
			return cbsStmt;
		}
		public void setCbsStmt(CBSStmt cbsStmt) {
			this.cbsStmt = cbsStmt;
		}
	}

	private static class StatementsForPrintAckPreparedStatementSetter implements PreparedStatementSetter {

		private List<ADFileRecord> adFileRecordL;

		private int nextIndex = 0;

		private String sqlBase;

		private String sqlMax;

		StatementsForPrintAckPreparedStatementSetter(List<ADFileRecord> adFileRecordL, String sqlBase) {
			this.adFileRecordL = adFileRecordL;
			this.sqlBase = sqlBase;
			sqlMax = sqlBase + (new SQLValuesList(BATCH_SELECT_MAX_SIZE)).toString();
		}

		public void setValues(PreparedStatement ps) throws SQLException {
			final int nextBatchSize = getNextBatchSize();
			for (int i = 1; i <= nextBatchSize; i++) {
				final ADFileRecord nextADFileRecord = adFileRecordL.get(nextIndex++);
				ps.setString(i, nextADFileRecord.getPatientAccount());
			}
		}

		/**
		 * @return The remaining number of records that have not been injected
		 *         into a statement yet.
		 */
		public int getRemainingCount() {
			return (nextIndex < adFileRecordL.size()) ? (adFileRecordL.size() - nextIndex) : 0;
		}

		/**
		 * Returns the parameterized query String for the next batch of records.
		 * 
		 * @param remainingCount
		 *            The remaining number of records to query.
		 * @return
		 */
		private String getNextBatchSelectSQL() {
			final int nextBatchSize = getNextBatchSize();

			String nextBatchSelectSQL;
			if (nextBatchSize >= BATCH_SELECT_MAX_SIZE) {
				nextBatchSelectSQL = sqlMax;
			} else {
				nextBatchSelectSQL = sqlBase + (new SQLValuesList(nextBatchSize)).toString();
			}

			return nextBatchSelectSQL;
		}

		public int getNextBatchSize() {
			return Math.min(getRemainingCount(), BATCH_SELECT_MAX_SIZE);
		}

	}

	@Override
	public CBSStmt get(long id) {
		final String sql = "SELECT * FROM CBSStmt WHERE id = " + id;

		final List<CBSStmt> cbsStmtL = jdbcTemplate.query(sql, new CBSStmtRowMapper());

		return (cbsStmtL.size() > 0) ? cbsStmtL.get(0) : null;
	}

	@Override
	public Long getCBSStmtId(long batchRunId, long accountId) {
		String sql = "SELECT id FROM CBSStmt WHERE batchRunId = ? AND accountId = ?";
		return jdbcTemplate.query(sql, new PreparedStatementSetter() {
			public void setValues(PreparedStatement preparedStatement) throws SQLException {
				preparedStatement.setLong(1, batchRunId);
				preparedStatement.setLong(2, accountId);
			}
		}, new ResultSetExtractor<Long>() {

			@Override
			public Long extractData(ResultSet rs) throws SQLException, DataAccessException {
				if (rs.next()) {
					return rs.getLong("id");
				}

				return null;
			}

		});

	}

	/**
	 * Maps a ResultSet to a new CBSStmt object.
	 * 
	 * @author Brad Pickle
	 *
	 */
	public static class CBSStmtRowMapper implements RowMapper<CBSStmt> {

		@Override
		public CBSStmt mapRow(ResultSet rs, int row) throws SQLException {
			final CBSStmt cbsStmt = new CBSStmt(rs.getLong("accountId"));

			cbsStmt.setId(rs.getLong("id"));
			cbsStmt.setReplacedStmtId(rs.getLong("replacedStmtId"));
			cbsStmt.setMessageId(rs.getLong("messageId"));
			cbsStmt.setAcntNumDisp(rs.getString("acntNumDisp"));
			cbsStmt.setAmountDue(new Money(rs.getDouble("amntDue"), 9));
			cbsStmt.setPrevBalance(new Money(rs.getDouble("preBalance"), 9));
			cbsStmt.setTotalCharges(new Money(rs.getDouble("totalCharges"), 9));
			cbsStmt.setTotalCredits(new Money(rs.getDouble("totalCredits"), 9));
			cbsStmt.setNewBalance(new Money(rs.getDouble("newBalance"), 9));
			cbsStmt.setLateStmtMsg(BooleanChar.from(rs.getString("lateStmtMsg")).isTrue());
			cbsStmt.setStatementDate(rs.getDate("statementDate"));
			cbsStmt.setProcessDate(rs.getDate("processDate"));
			cbsStmt.setBatchRunId(rs.getLong("batchRunId"));
			cbsStmt.setStatusId(rs.getInt("statusId"));
			cbsStmt.setPrintAckId(rs.getLong("printAckId"));
			cbsStmt.setPrintDate(rs.getDate("printDate"));
			cbsStmt.setCreatedBy(rs.getString("createdBy"));
			cbsStmt.setCreatedDate(rs.getDate("createdDate"));
			cbsStmt.setModifiedBy(rs.getString("modifiedBy"));
			cbsStmt.setModifiedDate(rs.getDate("modifiedDate"));

			return cbsStmt;
		}

	}

	@Override
	public Long getStatementCountWithStatus(int statusId) {
		final String sql = "SELECT COUNT(*) FROM CBSStmt WHERE statusId=" + statusId;

		final List<Long> results = jdbcTemplate.query(sql, new RowMapper<Long>() {
			@Override
			public Long mapRow(ResultSet rs, int row) throws SQLException {
				return new Long(rs.getLong(1));
			}
		});

		if ((results != null) && !results.isEmpty()) {
			return results.get(0);
		}

		return null;
	}

	@Override
	public int updateMessageIdForStatus(int statusId, long newMessageId) {
		final String sql = "UPDATE CBSStmt SET messageId=" + newMessageId + " WHERE statusId=" + statusId;
		return jdbcTemplate.update(sql);
	}

	@Override
	public int updateStatusForMessageId(int newStatusId, long messageId) {
		final String sql = "UPDATE CBSStmt SET statusId=" + newStatusId + " WHERE messageId=" + messageId;
		return jdbcTemplate.update(sql);
	}

	@Override
	public int updateAllStatusFromTo(final int fromStatus, final int toStatus) {
		final String sql = "UPDATE CBSStmt SET statusId=? WHERE statusId=?";
		return jdbcTemplate.update(sql, toStatus, fromStatus);
	}

	@Override
	public int updateStatusFromToForInitial(final int fromStatus, final int toStatus, final int initialStatus) {
		final String sql = "UPDATE CBSStmt SET statusId=? WHERE statusId=? and id in (select replacedstmtid from CBSStmt where statusid=? and replacedstmtid not in 0)";
		return jdbcTemplate.update(sql, toStatus, fromStatus, initialStatus);
	}

	@Override
	public void updatePrintAckFields(final List<CBSStmt> cbsStmtL, final long printAckId,
			final java.util.Date printDate, final int statusId, final String stationNum) {
		final String formattedPrintDate = new SimpleDateFormat("MM-dd-yyyy").format(printDate);

		final String sql = "UPDATE CBSStmt SET printAckId=" + printAckId + ", printDate=TO_DATE('" + formattedPrintDate
				+ "', 'MM-DD-YYYY'), statusId=" + statusId
				+ " WHERE id=? AND ((printAckId=null OR printDate=null OR statusId!=" + statusId
				+ ") OR id=(SELECT stmtId FROM CBSSiteStmt WHERE stmtId=CBSStmt.id AND isPrimary='Y' AND stationNum='"
				+ stationNum + "'))";

		jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

			@Override
			public void setValues(PreparedStatement ps, int i) throws SQLException {
				ps.setLong(1, cbsStmtL.get(i).getId());
			}

			@Override
			public int getBatchSize() {
				return cbsStmtL.size();
			}
		});
	}

	@Override
	public int[] undoPrintAcknowledgements(final List<Long> printAckRecIdList, final int sentStatusId) {
		if ((printAckRecIdList == null) || (printAckRecIdList.size() == 0))
			return new int[0];

		final String sql = "UPDATE CBSStmt SET statusId=" + sentStatusId
				+ ", printDate=NULL, printAckId=NULL WHERE printAckId=? ";

		final int[] rowCounts = new int[printAckRecIdList.size()];
		for (int i = 0; i < printAckRecIdList.size(); i++) {
			rowCounts[i] = jdbcTemplate.update(sql, printAckRecIdList.get(i));
		}

		return rowCounts;
	}

	@Override
	public int updateStatusFromToForPrintAckId(final int fromStatus, final int toStatus, final long printAckId) {
		final String sql = "UPDATE CBSStmt SET statusId=? WHERE statusId=? and printAckId=? ";
		return jdbcTemplate.update(sql, toStatus, fromStatus, printAckId);
	}

	@Override
	public int deleteByStatusId(final int statusId) {

		final String sql = "DELETE FROM CBSStmt where statusId = ?";
		return jdbcTemplate.update(sql, statusId);
	}

	public static String getUpdateCBSStmtSql() {
		return "update CBSStmt set statusId=? where id=?";
	}

	public static String getCBSSStmtJoinSql() {
		return "SELECT DISTINCT CBSSiteStmt.stationNum AS stationNum "
				+ "FROM CBSStmt INNER JOIN CBSSiteStmt ON CBSStmt.id = CBSSiteStmt.stmtId "
				+ "WHERE CBSSiteStmt.isPrimary = 'Y' and CBSStmt.messageId = ? and CBSStmt.statusId = ?";
	}

	// ToDo: Where Status is ACK sort by date and where clause for range of
	// dates
	@Override
	public List<CBSStmt> getListOfStmntsById(long accountid) {
		// Start Date is 5/1/xxxx to End Date 4/30/xxxx
		final String sql = "SELECT * FROM CBSStmt WHERE accountid = " + accountid
				+ " and (((sysdate < add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= add_months(trunc(sysdate, 'yy'), -12))) or ((sysdate >= add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= trunc(sysdate, 'yy')))) ORDER BY statementdate DESC";

		final List<CBSStmt> cbsStmtL = jdbcTemplate.query(sql, new CBSStmtRowMapper());
		return (cbsStmtL.size() > 0) ? cbsStmtL : null;
	}

	// Fix for Defect 616440
	// Yiping Yao - 11/15/2017, 01/03/2018
	// Using Starting / End Dates
	@Override
	public List<CBSStmt> getListOfStmntsById(long id, int delayDays, java.util.Date startDate, java.util.Date endDate)
	{
		if (startDate == null || endDate == null)
		{
			return null;
		}

		// SQL = SELECT * FROM CBSStmt
		//       WHERE accountId = id AND
		//           ( statusId = (SELECT ID FROM ProcStatus WHERE NAME = 'ACK') AND sysDate > (printDate + delayDays) ) AND
		//           ( 
		// ( ( (sysdate < add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= add_months(trunc(sysdate, 'yy'), -12)) ) or ( (sysdate >= add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= trunc(sysdate, 'yy')) ) )
		// Start Date is 5/1/xxxx to End Date 4/30/xxxx
		final String sql = "SELECT * FROM CBSStmt WHERE accountid = " + id
				+ " and (((sysdate < add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= add_months(trunc(sysdate, 'yy'), -12))) or ((sysdate >= add_months(trunc(sysdate, 'yy'), 4)) and (statementDate >= trunc(sysdate, 'yy')))) ORDER BY statementdate DESC";

		final List<CBSStmt> cbsStmtL = jdbcTemplate.query(sql, new CBSStmtRowMapper());
		return (cbsStmtL.size() > 0) ? cbsStmtL : null;
	}

	// Fix for Defect 616440
	// Yiping Yao - 11/15/2017, 01/03/2018
	// Using Starting Month and Total Months
	@Override
	public List<CBSStmt> getListOfStmntsById(long id, int delayDays, int startMonth, int totalMonths)
	{
		// SQL = SELECT * FROM CBSStmt
		//       WHERE accountId = id AND
		//           ( (statusId = (SELECT ID FROM ProcStatus WHERE NAME = 'ACK')) AND (sysDate > (printDate + delayDays)) ) AND
		//           ( ( (sysdate <  add_months(trunc(sysdate, 'yy'), startMonth)) AND (statementDate >= add_months(trunc(sysdate, 'yy'), -totalMonths)) ) OR
		//             ( (sysdate >= add_months(trunc(sysdate, 'yy'), startMonth)) AND (statementDate >= trunc(sysdate, 'yy')) ) )
		//       ORDER BY statementDate DESC;
		final String sql = "SELECT * FROM CBSStmt " +
						   " WHERE accountId = ? AND " +
						   "	 ( (statusId = (SELECT ID FROM ProcStatus WHERE NAME = 'ACK')) AND " +
						   "	   (sysDate > (printDate + ?)) ) AND " +
						   "	 ( ( (sysdate <  add_months(trunc(sysdate, 'yy'), ?)) AND " +
						   "		 (statementDate >= add_months(trunc(sysdate, 'yy'), -?)) ) OR " +
						   "	   ( (sysdate >= add_months(trunc(sysdate, 'yy'), ?)) AND " +
						   "		 (statementDate >= trunc(sysdate, 'yy')) ) ) " +
						   " ORDER BY statementDate DESC";

		final List<CBSStmt> cbsStmts = this.jdbcTemplate.query(sql, new PreparedStatementSetter()
				{
					@Override
		            public void setValues(PreparedStatement preparedStatement) throws SQLException
					{
						preparedStatement.setLong(1, id);
						preparedStatement.setInt(2, delayDays);
						preparedStatement.setInt(3, startMonth);
						preparedStatement.setInt(4, totalMonths);
						preparedStatement.setInt(5, startMonth);
					}
				}, new CBSStmtRowMapper() );

		return (cbsStmts != null && cbsStmts.size() > 0) ? cbsStmts : null;
	}
}
