package gov.va.cpss.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import javax.sql.DataSource;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import gov.va.cpss.dao.CBSMessageDAO;
import gov.va.cpss.dao.CBSSAuditDatesDAO;
import gov.va.cpss.model.cbs.CBSMessage;

/**
 * An implementation of the CBSMessageDAO interface.
 * 
 * @author Brad Pickle
 */
public class CBSMessageDAOImpl implements CBSMessageDAO {

	@SuppressWarnings("unused")
	private final Logger daoLogger;

	private NamedParameterJdbcTemplate namedParameterJdbcTemplate;

	private CBSSAuditDatesDAO cbssAuditDatesDAO;

	public CBSMessageDAOImpl() {
		daoLogger = Logger.getLogger(this.getClass().getCanonicalName());
	}

	public void setDataSource(DataSource dataSource) {
		this.namedParameterJdbcTemplate = new NamedParameterJdbcTemplate(dataSource);
	}

	public CBSSAuditDatesDAO getCbssAuditDatesDAO() {
		return cbssAuditDatesDAO;
	}

	public void setCbssAuditDatesDAO(CBSSAuditDatesDAO cbssAuditDatesDAO) {
		this.cbssAuditDatesDAO = cbssAuditDatesDAO;
	}

	@Override
	public List<CBSMessage> getCBSMessagesInStatus(int statusId) {
		final String sql = "SELECT * FROM CBSMessage WHERE statusId = " + statusId + " order by id";

		final List<CBSMessage> cbsMessages = namedParameterJdbcTemplate.query(sql, new RowMapper<CBSMessage>() {

			@Override
			public CBSMessage mapRow(ResultSet rs, int rowNum) throws SQLException {

				CBSMessage cbsMessage = new CBSMessage();

				cbsMessage.setId(rs.getLong("id"));
				cbsMessage.setBatchRunId(rs.getLong("batchRunId"));
				cbsMessage.setStatusId(rs.getInt("statusId"));
				cbsMessage.setFileName(rs.getString("fileName"));
				cbsMessage.setCreatedBy(rs.getString("createdBy"));
				cbsMessage.setCreatedDate(new java.util.Date(rs.getDate("createdDate").getTime()));
				cbsMessage.setModifiedBy(rs.getString("modifiedBy"));
				cbsMessage.setModifiedDate(new java.util.Date(rs.getDate("modifiedDate").getTime()));

				return cbsMessage;
			}
		});

		return cbsMessages;
	}

	@Override
	public CBSMessage get(long id) {
		final String sql = "SELECT * FROM CBSMessage WHERE id = " + id;

		final List<CBSMessage> cbsMessages = namedParameterJdbcTemplate.query(sql, new RowMapper<CBSMessage>() {

			@Override
			public CBSMessage mapRow(ResultSet rs, int rowNum) throws SQLException {

				CBSMessage cbsMessage = new CBSMessage();

				cbsMessage.setId(rs.getLong("id"));
				cbsMessage.setBatchRunId(rs.getLong("batchRunId"));
				cbsMessage.setStatusId(rs.getInt("statusId"));
				cbsMessage.setFileName(rs.getString("fileName"));
				cbsMessage.setCreatedBy(rs.getString("createdBy"));
				cbsMessage.setCreatedDate(new java.util.Date(rs.getDate("createdDate").getTime()));
				cbsMessage.setModifiedBy(rs.getString("modifiedBy"));
				cbsMessage.setModifiedDate(new java.util.Date(rs.getDate("modifiedDate").getTime()));

				return cbsMessage;
			}
		});

		return (cbsMessages.size() > 0) ? cbsMessages.get(0) : null;
	}

	@Override
	public void update(CBSMessage cbsMessage) {
		final String sql = "UPDATE CBSMessage " + "SET batchRunId=:batchRunId, statusId=:statusId, fileName=:fileName "
				+ "WHERE id=:id";

		namedParameterJdbcTemplate.update(sql, new BeanPropertySqlParameterSource(cbsMessage));

		// Load the new audit fields and set on cbsMessage
		cbssAuditDatesDAO.getAuditDates("CBSMessage", cbsMessage.getId(), cbsMessage);
	}

	@Override
	public long insert(CBSMessage cbsMessage) {
		// insert
		final String sql = "INSERT INTO CBSMessage (batchRunId, statusId, fileName) "
				+ "VALUES (:batchRunId, :statusId, :fileName)";
		final KeyHolder keyHolder = new GeneratedKeyHolder();

		namedParameterJdbcTemplate.update(sql, new BeanPropertySqlParameterSource(cbsMessage), keyHolder,
				new String[] { "id" });
		cbsMessage.setId(keyHolder.getKey().intValue());

		// Load the new audit fields and set on cbsMessage
		cbssAuditDatesDAO.getAuditDates("CBSMessage", cbsMessage.getId(), cbsMessage);

		return cbsMessage.getId();
	}

}
