package gov.va.cpss.dao.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Date;
//import java.util.ArrayList;
import java.util.List;
//import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.RowMapper;
//import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import gov.va.cpss.dao.APSDetailsDAO;

import gov.va.cpss.model.AITCDollar;
import gov.va.cpss.model.CBSSAbstractModel;

import gov.va.cpss.model.apps.APSDetails;
import gov.va.cpss.model.apps.APSPatient;
import gov.va.cpss.model.apps.APSSite;

/**
 * 
 * An implementation of the APSDetailsDAO interface.
 * 
 * Copyright HPE / VA
 * October 27, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
@SuppressWarnings({"nls", "static-method", "unchecked"})
public class APSDetailsDAOImpl extends CBSSBaseDAOImpl implements APSDetailsDAO
{
    public static final String TABLE_NAME = "APSDetails";

    @Override
	public List<APSDetails> getAllByAPSSite(final APSSite apsSite)
	{
		if (apsSite != null)
		{
			// Get all APSDetails ordered by ID that are in all APSPatient
			// associated with the specified APSSite.
			// SQL:       SELECT * FROM (SELECT * FROM APSDetails ORDER BY id) WHERE apsPatientId IN ( SELECT id FROM (SELECT * FROM APSPatient ORDER BY id) WHERE apsSiteId = 1 ) )
		    final String sql =   "SELECT *  FROM (SELECT * FROM " +                   TABLE_NAME + ORDER_BY_ID_CLAUSE + ") WHERE apsPatientId IN " +
		                       "( SELECT id FROM (SELECT * FROM " + APSPatientDAOImpl.TABLE_NAME + ORDER_BY_ID_CLAUSE + ") WHERE apsSiteId = " + apsSite.getId() + " )";

			List<APSDetails> detailsList = this.jdbcTemplate.query(sql, new RowMapper<APSDetails>()
			{
				@Override
				public APSDetails mapRow(ResultSet rs, int rowNum) throws SQLException
				{
				    return (APSDetails) mapResult(rs);
				}
			});

			return detailsList;
		}

		return null;
	}

    @Override
    public List<APSDetails> getByPatients(List<APSPatient> patients)
    {
        // Example SQ:
        // SELECT * FROM APSDetails WHERE apsPatientId IN (85, 86, 87, 88, 89, 90, 91);

        final String openingSQL = getSelectSQL() + " WHERE apsPatientId";

        // Extract patients ID's
        List<Long> ids = patients.stream().map(p -> Long.valueOf(p.getId())).collect(Collectors.toList());

        return (List<APSDetails>) batchSelectWithINClause(openingSQL, null, null, ids.toArray(), null);
    }

    @Override
    public int[] batchInsertWithoutKeys(List<Entry<APSPatient, APSDetails>> detailsList)
    {
/*
        final String sql = "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME + "(apsPatientId, datePosted, transAmnt, transAmntRaw) " +
                           "SELECT " + FIRST_ROWS_HINT + " " + APSPatientDAOImpl.TABLE_NAME + "." + ID + 
                           ", :datePosted, :transAmnt, :transAmntRaw " +
                           "FROM " + APSPatientDAOImpl.TABLE_NAME +
                           " WHERE icn = :icn AND dfn = :dfn AND oldAcntNum = :acntNum AND totAmntRcv = :totAmntRcv AND arAddressFlag = :arAddressFlag AND lastBillPrepDate = :lastBillPrepDate AND numOfPD = :numOfPD";

        List<Map<String, Object>> batchValues = new ArrayList<>(detailsList.size());

        for (Entry<APSPatient, APSDetails> detailsEntry : detailsList)
        {
            APSDetails details = detailsEntry.getValue();
            APSPatient patient = detailsEntry.getKey();

            batchValues.add(
                            new MapSqlParameterSource("datePosted", details.getDatePosted())
                                            .addValue("transAmnt", Double.valueOf(details.getTransactionAmount().getDouble()))
                                            .addValue("transAmntRaw", details.getTransactionAmountStr())
                                            .addValue("icn", patient.getIcn())
                                            .addValue("dfn", Long.valueOf(patient.getDfn()))
                                            .addValue("acntNum", patient.getAccountNumber())
                                            .addValue("totAmntRcv", Double.valueOf(patient.getTotalAmountReceived().getDouble()))
                                            .addValue("arAddressFlag", patient.getArAddressFlg())
                                            .addValue("lastBillPrepDate", patient.getLastBillPrepDate())
                                            .addValue("numOfPD", Integer.valueOf(patient.getNumOfPD()))
                       .getValues());
        }

        return this.namedParameterJdbcTemplate.batchUpdate(sql, batchValues.toArray(new Map[detailsList.size()]));
*/

        final String sql = "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME + "(apsPatientId, datePosted, transAmnt, transAmntRaw) " +
                           "SELECT " + FIRST_ROWS_HINT + " " + APSPatientDAOImpl.TABLE_NAME + "." + ID + 
                           ", ?, ?, ? " +
                           "FROM " + APSPatientDAOImpl.TABLE_NAME +
                           " WHERE icn = ? AND dfn = ? AND oldAcntNum = ? AND totAmntRcv = ? AND arAddressFlag = ? AND lastBillPrepDate = ? AND numOfPD = ?";
                           //ORDER_BY_ID_CLAUSE + " DESC ";

        return this.jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter()
        {
             @Override
             public void setValues(PreparedStatement ps, int i) throws SQLException
             {
                 Entry<APSPatient, APSDetails> atientDetails = detailsList.get(i);
                 mapRows(ps, atientDetails);
             }

             @Override
             public int getBatchSize()
             {
                 return detailsList.size();
             }
        });
    }

    /**
     * Map rows with patient and details.
     * 
     * @param ps
     * @param sitePatient
     * @throws SQLException
     */
    protected void mapRows(PreparedStatement ps, Entry<APSPatient, APSDetails> patientDetails) throws SQLException
    {
        APSPatient patient = patientDetails.getKey();
        APSDetails details = patientDetails.getValue();

        // Details
        ps.setDate(1, new Date(details.getDatePosted().getTime()));
        ps.setDouble(2, details.getTransactionAmount().getDouble());
        ps.setString(3, details.getTransactionAmountStr());

        // Patient
        ps.setString(4, patient.getIcn());
        ps.setLong(5, patient.getDfn());
        ps.setString(6, patient.getAccountNumber());
        ps.setDouble(7, patient.getTotalAmountReceived().getDouble());
        ps.setString(8, (patient.isArAddress()? "Y" : "N"));
        ps.setDate(9, (new Date(patient.getLastBillPrepDate().getTime())));
        ps.setInt(10, patient.getNumOfPD());
    }

    @Override
    public final String getTableName()
    {
        return TABLE_NAME;
    }

    @Override
    public final String getInsertSQL()
    {
        return "INSERT " + PARALLEL_HINT + " INTO " + TABLE_NAME + "(apsPatientId, datePosted, transAmnt, transAmntRaw) VALUES (?, ?, ?, ?)";
    }

    @Override
    protected void mapRows(PreparedStatement ps, CBSSAbstractModel model) throws SQLException
    {
        APSDetails details = (APSDetails) model;

        ps.setLong(1, details.getApsPatientId());
        ps.setDate(2, new Date(details.getDatePosted().getTime()));
        ps.setDouble(3, details.getTransactionAmount().getDouble());
        ps.setString(4, details.getTransactionAmountStr());
    }

    @Override
    protected CBSSAbstractModel mapResult(ResultSet rs) throws SQLException
    {
        APSDetails details = new APSDetails();

        details.setId(rs.getLong(ID));
        details.setApsPatientId(rs.getLong("apsPatientId"));
        details.setDatePosted((rs.getDate("datePosted") == null) ? null : new Date(rs.getDate("datePosted").getTime()));
        details.setTransactionAmount(new AITCDollar(rs.getDouble("transAmnt")));
        details.setTransactionAmountStr(rs.getString("transAmntRaw"));

        // Audit fields
        mapAuditFields(rs, details);

        return details;
    }

}
