package gov.va.cpss.dao;

import java.util.Date;
import java.util.List;
import java.util.Map;

import gov.va.cpss.model.cbs.CBSStmt;
import gov.va.cpss.model.printack.PrintAckRec;

/**
 * Data Access Object interface for the CBSStmt table.
 * 
 * @author Brad Pickle
 */
public interface CBSStmtDAO {

	/**
	 * Save the specified CBSStmt to the database.
	 * 
	 * @param statement
	 *            The statement to save. The id, createdBy, createdDate,
	 *            modifiedBy and modifiedDate properties will be updated to
	 *            match the database state after the insert.
	 * @return Auto-generated index ID of the entry in the database.
	 */
	public long save(CBSStmt statement);

	/**
	 * Batch save the list of CBSStmts to the database. Unlike the save()
	 * method, none of the fields populated by the database will be updated (id
	 * will not be set).
	 * 
	 * @param statements
	 *            The list of statements to save
	 */
	public void saveBatch(List<CBSStmt> statements);

	/**
	 * Update all statements from specified status to specified status.
	 * 
	 * @param fromStatus
	 *            The statements of this status to update.
	 * @param toStatus
	 *            The final status to change the statements to.
	 * @return Number of rows updated.
	 */
	public int updateAllStatusFromTo(final int fromStatus, final int toStatus);

	/**
	 * Update statements from specified status to specified status for specified
	 * initial status.
	 * 
	 * @param fromStatus
	 *            The statements of this status to update.
	 * @param toStatus
	 *            The final status to change the statements to.
	 * @param initialStatus
	 *            The value for initial status.
	 * @return Number of rows updated.
	 */
	public int updateStatusFromToForInitial(final int fromStatus, final int toStatus, final int initialStatus);

	/**
	 * Delete records having the specified status ID.
	 * 
	 * @param statusId
	 *            The status ID.
	 * @return Number of rows deleted.
	 */
	public int deleteByStatusId(final int statusId);

	/**
	 * Get an existing CBSStmt from the database by the specified account number
	 * and year/month.
	 * 
	 * @param accountNumber
	 *            The account number to query.
	 * @param statementDate
	 *            The year/month date to query.
	 * @return The existing CBSStmt or null if not found.
	 */
	public CBSStmt getExistingStatementByAccountAndMonth(final long accountNumber, final Date statementDate);

	/**
	 * Get the statusId of the latest existing CBSStmt from the database by the
	 * specified account number and year/month.
	 * 
	 * @param accountNumber
	 *            The account number to query.
	 * @param statementDate
	 *            The year/month date to query.
	 * @return The statusId of the existing CBSStmt or null if not found.
	 */
	Integer getStatusIdForExistingStatementByAccountAndMonth(long accountNumber, Date statementDate);

	/**
	 * Get the number of rows in the CBSStmt table with the given statusId
	 * 
	 * @param statusId
	 *            statusId of rows to count
	 * @return Number of rows with given statusId
	 */
	Long getStatementCountWithStatus(int statusId);

	/**
	 * Update all of the CBSStmt records having the given statusId with the
	 * given messageId.
	 * 
	 * @param statusId
	 *            Records with this statusId will be updated
	 * @param newMessageId
	 *            The new messageId
	 * @return Number of rows updated
	 */
	int updateMessageIdForStatus(int statusId, long newMessageId);

	/**
	 * Update all of the CBSStmt records having the given messageId with the
	 * given statusId.
	 * 
	 * @param newStatusId
	 *            The new statusId
	 * @param messageId
	 *            Records with this messageId will be updated
	 * @return Number of rows updated
	 */
	int updateStatusForMessageId(int newStatusId, long messageId);

	/**
	 * Get the List of consolidated statements (CBSStmt) for the given
	 * CBSMessage messageId and primary site designated by stationNum.
	 * 
	 * @param cbsMessageId
	 *            return CBSStmt records with this messageId
	 * @param stationNum
	 *            return CBSStmt records with a primary station designated by
	 *            this station number
	 * @return List<CBSStmt> for all statements for the given messageId and
	 *         stationNum
	 */
	List<CBSStmt> getMessageStatementsForSite(long cbsMessageId, String stationNum);

	/**
	 * Get CBSStmt with the given id
	 * 
	 * @param id
	 *            id of the CBSStmt to get
	 * @return CBSStmt with given id
	 */
	CBSStmt get(long id);

	/**
	 * Get a List of CBSStmt with the given id
	 * 
	 * @param id
	 *            id of the CBSStmt to get
	 * @return List CBSStmt with given id
	 */
	List<CBSStmt> getListOfStmntsById(long id);

	// Fix for Defect 616440
	// Yiping Yao - 11/15/2017, 01/03/2018
	/**
	 * Get a List of CBSStmt with the given id and between Start Date and End Date
	 * 
	 * @param id - id of the CBSStmt to get
	 * @param delayDays - delay days
	 * @param startDate - start date of statement date range
	 * @param endDate - end date of statement date range
	 * @return List CBSStmt with given id
	 */
	List<CBSStmt> getListOfStmntsById(long id, int delayDays, Date startDate, Date endDate);

	// Fix for Defect 616440
	// Yiping Yao - 11/15/2017, 01/03/2018
	/**
	 * Get a List of CBSStmt with the given id and between Start Date and End Date
	 * 
	 * @param id - id of the CBSStmt to get
	 * @param delayDays - delay days
	 * @param startMonth - start month of statement date range (1 - January, 2 - February, 3 - Match, 4 - April, ...)
	 * @param totalMonths - total number of months for statement date range
	 * @return List CBSStmt with given id
	 */
	List<CBSStmt> getListOfStmntsById(long id, int delayDays, int startMonth, int totalMonths);

	/**
	 * Get the id of the CBSStmt with the given batchRunId and accountId
	 * 
	 * @param batchRunId
	 * @param accountId
	 * @return Id of matching CBSStmt or 0 if not found.
	 */
	Long getCBSStmtId(long batchRunId, long accountId);

	/**
	 * Get statements for each of the patient accounts acknowledged by the given
	 * printAckRec for the given statement date. Only statements with SENT and
	 * ACK statuses will be returned.
	 * 
	 * @param printAckRec
	 *            return statements for patient accounts in this PrintAckRec's
	 *            printAckDetailL list.
	 * @param statementDate
	 *            return statements with this statement date.
	 * @param sentStatusId
	 *            the id of the SENT status
	 * @param ackStatusId
	 *            the id of the ACK status
	 * @return Map<String, CBSStmt> for the given PrintAckRec and statement date
	 *         with the patient's oldAcntNum as keys
	 */
	Map<String, CBSStmt> getStatementsForPrintAck(PrintAckRec printAckRec, Date statementDate, int sentStatusId,
			int ackStatusId);

	/**
	 * Updates the printAckId, printDate and statusId fields for all CBSStmt
	 * objects in the given List.
	 * 
	 * @param cbsStmtL
	 *            List of CBSStmt objects to update.
	 * @param printAckId
	 *            Update with this printAckId.
	 * @param printDate
	 *            Update with this printDate.
	 * @param statusId
	 *            Update with this statusId.
	 */
	void updatePrintAckFields(List<CBSStmt> cbsStmtL, long printAckId, Date printDate, int statusId, String stationNum);

	/**
	 * Updates all CBSStmt rows with the given fromStatus and printAckId to the
	 * given toStatus.
	 * 
	 * @param fromStatus
	 *            CBSStmt rows with this fromStatus will be updated.
	 * @param toStatus
	 *            CBSStmt rows updated with this new status.
	 * @param printAckId
	 *            CBSStmt rows with this printAckId will be updated.
	 * @return The number of CBSStmt rows updated.
	 */
	int updateStatusFromToForPrintAckId(int fromStatus, int toStatus, long printAckId);

	/**
	 * Updates all CBSStmt rows with printAckIds matching the ids of the given
	 * List of PrintAckRec record ids, setting the statusId to the given
	 * sentStatusId, and setting the printAckId and printDate fields to null.
	 * Intended to be used to undo statements that have been print acknowledged
	 * and revert back to the SENT status.
	 * 
	 * @param printAckRecIdList
	 *            List of PrintAckRec record ids to be undone. The ids will be
	 *            used to identify CBSStmt rows with corresponding printAckIds.
	 * @param sentStatusId
	 *            SENT status id to set matching CBSStmt rows to.
	 * @return An array of ints with each position representing the number of
	 *         rows updated for the corresponding PrintAckRec record id in the
	 *         same position in printAckRecIdList.
	 */
	int[] undoPrintAcknowledgements(List<Long> printAckRecIdList, int sentStatusId);

}
