package gov.va.cpss.dao;

import java.util.List;

import gov.va.cpss.model.cbs.CBSSiteStmt;
import gov.va.cpss.model.cbs.CBSStmt;

public interface CBSSiteStmtDAO {

	/**
	 * Save the specified CBSSiteStmt to the database.
	 * 
	 * @param siteStatement
	 *            The statement to save. The id, createdBy, createdDate,
	 *            modifiedBy and modifiedDate properties will be updated to
	 *            match the database state after the insert if updateAuitDates
	 *            is true.
	 * @param updateAuditDates
	 *            Update the audit dates after inserting.
	 * @return Auto-generated index ID of the entry in the database.
	 */
	public long save(CBSSiteStmt siteStatement, final boolean updateAuditDates);

	/**
	 * Save the specified CBSSiteStmt to the database, using the given
	 * batchRunId and accountId to identify the id of the corresponding CBSStmt
	 * record. For performance reasons, the audit fields will not be updated.
	 * 
	 * @param siteStatement
	 *            The statement to save.
	 * @return Auto-generated index ID of the entry in the database.
	 */
	long save(CBSSiteStmt siteStatement, long batchRunId, long accountId);

	/**
	 * Batch save the nested CBSSiteStmts contained within the list of CBSStmts
	 * to the database. Unlike the save() method, none of the fields populated
	 * by the database will be updated (id will not be set). The accountId and
	 * batchRunId from the corresponding CBSStmt are used to identify the
	 * CBSStmt record id, rather than taking the id from the CBSStmt object.
	 * This is for performance reasons.
	 * 
	 * @param statements
	 *            The list of statements to save
	 */
	void saveBatch(List<CBSStmt> statements);

	/**
	 * Get the List of consolidated statement sites (CBSSiteStmt) for the given
	 * CBSMessage messageId and primary site designated by stationNum. Also
	 * loads the site patient.
	 * 
	 * @param cbsMessageId
	 *            return CBSSiteStmt records for CBSStmt records with this
	 *            messageId
	 * @param stationNum
	 *            return CBSSiteStmt records for CBSStmt records with a primary
	 *            station designated by this station number
	 * @return List<CBSSiteStmt> for all statements for the given messageId and
	 *         stationNum
	 */
	List<CBSSiteStmt> getMessageStatementSitesForSite(long cbsMessageId, String stationNum);

	/**
	 * Get the List of consolidated statement sites (CBSSiteStmt) for the given
	 * CBSStmt stmtId. Also load the site patient if specified by
	 * loadSitePatient.
	 * 
	 * @param stmtId
	 *            return CBSSiteStmt records for CBSStmt records with this
	 *            CBSStmt id
	 * @param loadSitePatient
	 *            if true, load the CBSSitePatient for each CBSSiteStmt
	 * @return List<CBSSiteStmt> for all statements for the given stmtId
	 */
	List<CBSSiteStmt> getAllByCBSStmtID(long stmtId, boolean loadSitePatient);

}
