package gov.va.cpss.dao;

import java.util.List;

import gov.va.cpss.model.CBSSAbstractModel;
import gov.va.cpss.model.ProcessStatus.Status;

/**
 * 
 * A generic DAO interface for CBSS.
 * 
 * Copyright HPE / VA
 * November 11, 2016
 * 
 * @author Yiping Yao
 * @version 1.0.0
 *
 */
public interface CBSSBaseDAO
{
    /**
     * Get the last index from the underlying table.
     */
    public Long getLastIndex();

    /**
     * Save CBSSAbstractModel and return the generated key.
     * 
     * @param model
     * @return the generated key
     * 
     */
    public long save(final CBSSAbstractModel model);

    /**
     * Batch insert and save a list of CBSSAbstractModel and return the generated keys.
     * 
     * @param models
     * @return the generated keys
     * 
     */
    public long[] batchSave(final List<? extends CBSSAbstractModel> models);

    /**
     * Update the status of the model.
     * 
     * @param model
     * @return the number of updated models.
     */
    public int updateStatus(final CBSSAbstractModel model);

    /**
     * Update the status based on the model key / primary ID.
     * 
     * @param id
     * @param status
     * @return the number of updated models.
     */
    public int update(final long id, final Status status);

    /**
     * Batch update the status of list of model keys / primary ID's.
     * 
     * @param ids
     * @param status
     * @return the numbers of updated models for each ID.
     */
    public int[] update(final long[] ids, final Status status);

    /**
     * Delete a CBSSAbstractModel by its primary ID / key.
     * 
     * @param inID
     * @return the number of deleted model
     * 
     */
    public int deleteById(final long inID);

    /**
     * Select data rows.
     * 
     * @param whereClause
     * @param orderByClause
     * @return list of selected models
     * 
     */
    public List<? extends CBSSAbstractModel> select(final String whereClause, final String orderByClause);

    /**
     * Select data rows with pagination.
     * 
     * @param pageNumber
     * @param pageSize
     * @return list of selected data rows
     * 
     */
    public List<? extends CBSSAbstractModel> selectWithPaging(int pageNumber, int pageSize);

    /**
     * Batch select data based on the input parameters and using the IN clause.
     * 
     * @param openingSQL
     * @param closingSQL
     * @param openingParams
     * @param inClauseParams
     * @param closingParams
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelectWithINClause(final String openingSQL, final String closingSQL,
                                                                     final Object[] openingParams, final Object[] inClauseParams, final Object[] closingParams);
    /**
     * Batch query database based on the array of single parameter.
     * 
     * @param sql
     * @param params
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelect(final String sql, final Object[] params);

    /**
     * Batch query database based on the arrays of multiple parameters.
     * 
     * @param sql
     * @param list of params
     * @return list of models
     * 
     */
    public List<? extends CBSSAbstractModel> batchSelect(final String sql, final List<List<?>> paramsList);

    /**
     * Batch insert / save without returning the generated keys.
     *
     * @param models
     * @return int[] - an array of the number of rows affected by each statement
     * 
     */
    public int[] batchInsert(final List<? extends CBSSAbstractModel> models);

}
