package gov.va.cpss.dao;

import java.util.Date;
import java.util.List;

import gov.va.cpss.model.ProcessStatus.Status;
import gov.va.cpss.model.apps.APSStmt;
import gov.va.cpss.model.appsprintack.APPSADFileRecord;

/**
 * Data Access Object interface for the APSStmt table.
 * 
 * Copyright HPE / VA
 * January 17, 2017
 * 
 * @author Yiping Yao
 * @version 1.0.0
 * 
 */
public interface APSStmtDAO extends CBSSBaseDAO
{
	/**
	 * Get the number of rows in the APSStmt table with the given generate process id and status.
	 * 
	 * @param status
	 *            status of rows to count
	 * @return Number of rows with given status
	 * 
	 */
	public Long getGenCountWithStatus(final long genProcessId, final Status status);

	/**
	 * Get existing Statements.
	 * 
	 * @param inStmts
	 * @return
	 * 
	 */
	public List<APSStmt> getExistingStatements(final List<APSStmt> inStmts);

	/**
	 * Batch update a list of Statements.
	 * 
	 * @param inStmts
	 * @return
	 * 
	 */
    public int[] batchUpdate(final List<APSStmt> inStmts);

    /**
     * Delete Statements with specified Generate APPS process Id.
     *  
     * @param processId
     * 
     */
    public void deleteByGenProcessId(final long processId);

    /**
     * Update Statement status with specified Generate APPS process Id.
     * 
     * @param processId
     * @param status
     * @return
     * 
     */
    public int updateStatusByGenProcessId(final long processId, final Status status);

    /**
     * Get APSStmt with the given id
     * 
     * @param id
     *            id of the APSStmt to get
     * @return APSStmt with given id
     */
    APSStmt get(long id);

    /**
     * Get statements for each of the patient accounts acknowledged by the given
     * printAckRec for the given statement date. Only statements with SENT and
     * ACK statuses will be returned.
     * 
     * @param printAckDetailL
     *            return statements relating to these AD records
     * @param statementYear
     *            return statements with this statement year.
     * @param sentStatusId
     *            the id of the SENT status
     * @param ackStatusId
     *            the id of the ACK status
     * @return List<APSStmt> for the given PrintAckRec and statement date. The
     *         statements will be ordered in the List based on the order of the
     *         patient accounts in the printAckDetailL list. Only one statement
     *         will be returned for each account (indeterminate). If a statement
     *         is not found for a given patient account, then a null will be
     *         returned in its place in the List.
     */
    List<Long> getStatementIdsForPrintAck(final List<APPSADFileRecord> printAckDetailL, final int statementYear,
			final int sentStatusId, final int ackStatusId, final Date printDate, final long batchRunProcessId);

    /**
     * Updates the batchRunProcessIdAck, printDate and statusId fields for all APSStmt
     * objects in the given List.
     * 
     * @param APSStmtL
     *            List of APSStmt objects to update.
     * @param batchRunProcessIdAck
     *            Update with this printAckId.
     * @param printDate
     *            Update with this printDate.
     * @param statusId
     *            Update with this statusId.
     */
    void updatePrintAckFields(List<Long> apsStmtIdL, long batchRunProcessIdAck, Date printDate, int statusId);

    /**
     * Updates all APSStmt rows with batchRunProcessIdAcks matching the ids of the given
     * List of BatchRunProcess record ids, setting the statusId to the given
     * sentStatusId, and setting the batchRunProcessIdAck and printDate fields to null.
     * Intended to be used to undo statements that have been print acknowledged
     * and revert back to the SENT status.
     * 
     * @param batchRunProcessIdAckList
     *            List of BatchRunProcess record ids to be undone. The ids will be
     *            used to identify APSStmt rows with corresponding batchRunProcessIdAcks.
     * @param sentStatusId
     *            SENT status id to set matching APSStmt rows to.
     * @return An array of ints with each position representing the number of
     *         rows updated for the corresponding BatchRunProcess record id in the
     *         same position in batchRunProcessIdAckList.
     */
    int[] undoPrintAcknowledgements(List<Long> batchRunProcessIdAckList, int sentStatusId);
    
    /**
     * Get the List of annual statements (APSStmt) for the given
     * appsMessageId.
     * 
     * @param status
     *            return APSStmt records with this status
     */ 
    public List<APSStmt> getAllStatementsForStatus(final Status status);
    
    /**
     * Updates the statusId to SENT and the batchRunProcessIdSend to the current batch process id for a
     * given set of APSStmts.
     * @param apsStmtL list of APSStmts to update
     * @param batchRunProcessId current batch run process id
     * @return rows updated per statement
     */
    public int[] updateStatusToSent(final List<APSStmt> apsStmtL, final long batchRunProcessId);
    
    /**
     * Gets a paged list of annual statements.
     * @param page index of the desired page
     * @param pageSize size of the page
     * @param statementYear retrieve statements from this year
     * @param status retrieve statements with this status
     * @return list of APSStmts from statementYear with status, length no greater than pageSize
     */
    public List<APSStmt> getApsStmtsWithPaging(final int page, final int pageSize, final int statementYear, final Status status);
}
