package gov.va.cpss.controller;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import gov.va.cpss.model.BatchItem;
import gov.va.cpss.model.BatchJob;
import gov.va.cpss.model.BatchRun;
import gov.va.cpss.model.BatchStatus;
import gov.va.cpss.model.TestTime;
import gov.va.cpss.security.AuthorityUtils;
import gov.va.cpss.service.BatchService;
import gov.va.cpss.service.CbssAccess;
import gov.va.cpss.service.SchedulingService;
import gov.va.cpss.util.CpssUtils;

import static gov.va.cpss.ESAPI.EsapiValidator.validateStringInput;
import static gov.va.cpss.ESAPI.EsapiValidationType.CROSS_SITE_SCRIPTING_PERSISTENT;
import static gov.va.cpss.ESAPI.EsapiValidationType.SAFE_STRING;
/**
 * Handles requests for the application home page.
 */
@Controller
public class CPSSController {
	
	private static final Logger httpLogger = Logger.getLogger(CPSSController.class.getCanonicalName());
		
	private BatchService batchService;
	private SchedulingService schedulingService;

	public SchedulingService getSchedulingService() {
		return schedulingService;
	}

	public void setSchedulingService(SchedulingService schedulingService) {
		this.schedulingService = schedulingService;
	}

	public BatchService getBatchService() {
		return batchService;
	}

	public void setBatchService(BatchService batchService) {
		this.batchService = batchService;
	}


	// Root home page.
	// Requires no special privileges.
	@RequestMapping(value = "/", method = RequestMethod.GET)
	public ModelAndView home(ModelAndView model,@RequestHeader Map<String,String> headers) throws IOException {
		
			httpLogger.info( "CBSS_Access:");
			httpLogger.info( " TimpeStamp :  " + new Date().toString());
			httpLogger.info( " Header Info :  " );
			for (String elem: headers.keySet()) { 
				httpLogger.info(elem + " : " + headers.get(elem)); 
			}
			
			httpLogger.info("authority   " + hasAuthority(AuthorityUtils.AuthorityEnum.ROLE_ADMIN));
			//This is for 2FA  accesss check
			
			if(this.getPrincipal().equalsIgnoreCase(CbssAccess.getCbssAccessGlobal().trim())){
				return new ModelAndView("redirect:/batch");
			}else{
				httpLogger.info( " Access Denied for this user Role: " + this.getPrincipal());
				return new ModelAndView("redirect:/denied");
			
			}
			
	}

	// Display the list of batch processes and their current status.
	// Requires operator to have ADMIN privileges.
	@SuppressWarnings("null")
	@RequestMapping(value = "/batch")
	public ModelAndView batch(ModelAndView model) {
		// Get the latest list of batch items from the Batch Service.
		List<BatchItem> batchItemList = batchService.getCBSSLatestBatchItemsList();
		List<BatchItem> validatedBatchItemList = new ArrayList<BatchItem>();
		
		for (BatchItem batchItem : batchItemList) {
				
				BatchItem validatedBatchItem = new BatchItem(validateBatchJob(batchItem.getJob()) , batchItem.getRun());
				validatedBatchItemList.add(validatedBatchItem);
				
		}
		
		model.addObject("batchItemL", validatedBatchItemList );
		model.addObject("version", CpssUtils.getVersion());
		model.addObject("timeoutGlobalUrl", CbssAccess.getCbssAccessRedirectUrl());
		model.setViewName("batch");
		return model;
	}


	// JSON ajax to update the main cpss batch html table.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "batch/updateJobStatusTable", method = RequestMethod.GET, headers = "Accept=application/json")
	public @ResponseBody List<BatchItem> updateJobStatusTable() {
		// Get most recent run of the batch jobs.
		return batchService.getCBSSLatestBatchItemsList();
	}

	// JSP ajax to populate the batch configuration dialog.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "batch/configDialog", params = { "id" }, method = RequestMethod.GET)
	public ModelAndView configDialog(@RequestParam(value = "id", required = true) int id, ModelAndView model) {
		// Get the list of the batch types to populate the radio buttons.
		model.addObject("rts", batchService.getBatchTypeList());

		// Get the desired batch job by id.
		model.addObject("batchJob", batchService.getBatchJobById(id));

		model.setViewName("batch/BatchConfigDialog");
		return model;
	}

	// JSP ajax to populate the batch run dialog.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "batch/runDialog", params = { "id" }, method = RequestMethod.GET)
	public ModelAndView runDialog(@RequestParam(value = "id", required = true) int id, ModelAndView model) {
		// Get the desired batch job by id.
		BatchItem bi = batchService.getLatestBatchItemById(id);
		if ((bi.getRun() != null) && bi.getRun().getBatchStatus().getJobStatus() == BatchStatus.JobStatus.RUNNING) {
			model.addObject("batchJob", null);
		} else {
			model.addObject("batchJob", bi.getJob());
		}

		model.setViewName("batch/BatchRunDialog");
		return model;
	}

	
	@RequestMapping(value = "/batchStatus/{id}", method = RequestMethod.GET)
	public ModelAndView batchStatus(@PathVariable(value ="id")int id, ModelAndView model) {
		// Get the desired batch job by id.
		
		model.addObject("batchItem", validatedBatchItem(batchService.getLatestBatchItemById(id)));
		
		
		model.setViewName("batchStatus");

		return model;
	}

	// Handle saving of batch updates.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "batch/saveBatch", method = RequestMethod.POST)
	public ModelAndView saveBatch(@ModelAttribute BatchJob batchJob) {
		schedulingService.saveBatchJob(batchJob);
		return new ModelAndView("redirect:/batch");
	}
		
	// Manually run the specified job.
	// Requires operator to have ADMIN privileges.
	@RequestMapping(value = "batch/runJob", method = RequestMethod.POST)
	public ModelAndView runJob(@ModelAttribute BatchJob batchJob) {
		// Use the Scheduling Service to run the job.
		schedulingService.runJob(batchJob.getId());

		return new ModelAndView("redirect:/batch");
	}

	// JSON ajax to update the time.
	// Requires no special privileges.
	@RequestMapping(value = "/updateTime", method = RequestMethod.GET, headers = "Accept=application/json")
	public @ResponseBody TestTime updateTime() {

		return new TestTime();
	}

	// Statement Placeholder.
	// Requires operator to have USER privileges.
	@RequestMapping(value = "/statement", method = RequestMethod.GET)
	public ModelAndView statementPage(ModelAndView model) {

		// Check for authority and if not, then redirect to denied.
		if (!hasAuthority(AuthorityUtils.AuthorityEnum.ROLE_USER)) {
			return new ModelAndView("redirect:/denied");
		}

		model.addObject("user", getPrincipal());
		model.setViewName("statement");
		return model;
	}

	// Access denied page.
	// Requires no special privileges.
	@RequestMapping(value = "/denied", method = RequestMethod.GET)
	public ModelAndView deniedPage(ModelAndView model) {
		model.addObject("user", getPrincipal());
		model.setViewName("denied");
		return model;
	}

	// Logout page.. 
	
	@RequestMapping(value = "/logout", method = RequestMethod.GET)
	public  ModelAndView logoutPage(ModelAndView model, HttpServletRequest request, HttpServletResponse response) {
		Date date = new Date();
		httpLogger.info("TimeStamp : Logout : " + date.toString());
		httpLogger.info("User role Logged out : " + this.getPrincipal()); 
		SecurityContextHolder.getContext().getAuthentication().setAuthenticated(false);
		httpLogger.info("session : reidirect to logout success"  );
		return new ModelAndView("redirect:/logoutSuccess");			
					
	}
	
	// Logout success -user is redirected to the IAM Authenticated Landing Page.
	// Requires no special privileges.
		@RequestMapping(value = "/logoutSuccess", method = RequestMethod.GET)
		public  ModelAndView logoutSuccessPage(ModelAndView model, HttpServletRequest request, HttpServletResponse response) {
			 HttpSession session = request.getSession(false);
			httpLogger.info("Logged out success :" );
			
			if(session != null){	
				httpLogger.info("session is valid : So invalidate before logout" );
				response.setHeader("Cache-Control", "no-cache, no-store, must-revalidate");
				response.setHeader("Pragma", "no-cache");
				response.setDateHeader("max-age", 0);
				response.setDateHeader("Expires", 0);
				httpLogger.info("session invalidate before: "+ session.getId() );
			    session.invalidate();
				httpLogger.info("session invalidate after: "+ session.toString() );
				httpLogger.info("session invalidate after: "+ session.getId());
			}
			return new ModelAndView("redirect:"+CbssAccess.getCbssAccessRedirectUrl());		
						
		}
		
	// Get the operator name.
	private String getPrincipal() {
		String userName = null;
		Object principal = SecurityContextHolder.getContext().getAuthentication().getPrincipal();

		if (principal instanceof UserDetails) {
			userName = ((UserDetails) principal).getUsername();
		} else {
			userName = principal.toString();
		}
		return userName;
	}

	// Check the authorization role of the operator.
	private boolean hasAuthority(final AuthorityUtils.AuthorityEnum role) {
		Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		Collection<? extends GrantedAuthority> authorities = authentication.getAuthorities();

		return authorities.contains(new SimpleGrantedAuthority(role.getAuthority()));
	}
	
	public BatchItem validatedBatchItem(BatchItem batchItem)
	{
		BatchItem validatedBatchItem = new BatchItem(validateBatchJob(batchItem.getJob()), batchItem.getRun());
		return validatedBatchItem; 
	}

	public BatchJob validateBatchJob(BatchJob job) {
		BatchJob validatedBatchJob = new BatchJob();
		validatedBatchJob.setDescription(validateStringInput(job.getDescription(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setEmail(validateStringInput(job.getEmail(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setName(validateStringInput(job.getName(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setId(Integer.valueOf(validateStringInput(Integer.toString(job.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchJob.setSchedule(validateStringInput(job.getSchedule(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchJob.setTypeId(Integer.valueOf(validateStringInput(Integer.toString(job.getTypeId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		return validatedBatchJob;
	}
	
	public BatchRun validateBatchRun(BatchRun run)
	{
		BatchRun validatedBatchRun = new BatchRun();
		validatedBatchRun.setBatchStatus(validateBatchStatus(run.getBatchStatus()));
		validatedBatchRun.setEndDate(run.getEndDate());
		validatedBatchRun.setId(Integer.valueOf(validateStringInput(Integer.toString(run.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchRun.setJobId(Integer.valueOf(validateStringInput(Integer.toString(run.getJobId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchRun.setMessage(validateStringInput(run.getMessage(), SAFE_STRING));
		validatedBatchRun.setStartDate(run.getStartDate());
		validatedBatchRun.setStatusId(Integer.valueOf(validateStringInput(Integer.toString(run.getStatusId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		
		return validatedBatchRun;
		
	}
	
	public BatchStatus validateBatchStatus(BatchStatus status)
	{
		BatchStatus validatedBatchStatus = new BatchStatus();
		validatedBatchStatus.setDescription(validateStringInput(status.getDescription(), CROSS_SITE_SCRIPTING_PERSISTENT));
		validatedBatchStatus.setId(Integer.valueOf(validateStringInput(Integer.toString(status.getId()), CROSS_SITE_SCRIPTING_PERSISTENT)));
		validatedBatchStatus.setJobStatus(status.getJobStatus());
		validatedBatchStatus.setName(validateStringInput(status.getName(), CROSS_SITE_SCRIPTING_PERSISTENT));
		
		return validatedBatchStatus;
	}

	

}
